#include "rp2mag_encode.h"
#include "debug_utils.h"
#include "utils.h"
#include <stdlib.h>
#include <string.h>

#define MAX_4_BIT_CHARS 138
#define MAX_5_BIT_CHARS 107
#define MAX_6_BIT_CHARS 92
#define MAX_7_BIT_CHARS 79
#define MAX_8_BIT_CHARS 72

#define MAG_END_SENTINEL '?'
#define JIS2_SENTINEL    '\x7f' // DEL character

#define MAG_BUFSIZE 112

#define START_OFFSET_5_BIT 48
#define START_OFFSET_7_BIT 32

static char mag_5[16] = {0};
static char mag_7[80] = {0};

/******************************************************************************

                       pack_bits()

  Packs the bits from encoded bytes to a continuous bitstream.

  parameters: in_data - pointer to the input data buffer
                        containing the encoded bytes
              in_count - the number of encoded chars
              enc_sz - Size of the endcoded characters in the input data.
              out_data - Pointer to the output buffer that will
                         contain the packed bitstream
              lead_count - count of the number of leading zeros to add
              trail_cont - count of the number of trailing zeros to add

  Returns: The total number of bits packed into the buffer

******************************************************************************/
static int32_t pack_bits(const uint8_t *in_data,
                         const int      in_count,
                         const int      enc_sz,
                         uint8_t *      out_data,
                         const int      lead_count,
                         const int      trail_count) {
    int  out_bit = 7;
    char val     = 0;
    int  i       = 0;
    int  j       = 0;

    // Append leading 0s
    for (i = 0; i < lead_count; i++) {
        *out_data &= 0;
        out_bit--;

        if (out_bit < 0) {
            out_bit = 7;
            out_data++;
        }
    }

    // Pack the bits into the output buffer
    // Loop over the number of characters
    for (i = 0; i < in_count; i++) {
        // Pack each bit individually
        for (j = 7; j > (7 - enc_sz); j--) {
            val = (in_data[i] >> j) & 0x01;
            val <<= (out_bit--);

            *out_data |= val;

            if (out_bit < 0) {
                out_bit = 7;
                out_data++;
            }
        }
    }
    return ((in_count * enc_sz) + lead_count + trail_count);
}

/******************************************************************************

                       mag_encoder_setup_5bit_data()

 This will take a pointer to a string of MAG data for 5 bpc and use a
 look up table to convert into the correct 5 bit with parity bit
 reversed bytes and place these into the output string.
 Onto the end of this is placed the LRC and a terminating NULL

  Input parameters: void

  Returns:          the number of bits encoded

******************************************************************************/
int32_t mag_encoder_setup_5bit_data(char *             input_data,
                                    uint8_t *          output_data,
                                    const int          lead,
                                    const bool         bAddEndSentinel) {
    int     n          = 0;
    int     bit        = 0;
    char    xor_result = 0;
    char    temp       = 0;
    uint8_t parity     = 0xFF;

    uint8_t tmp_buf[MAG_BUFSIZE] = {0};

    /* If first character is a 'space' then skip */
    if (*input_data == ' ') {
        input_data++;
    }

    while (*input_data && n < MAX_5_BIT_CHARS) {
        tmp_buf[n] = mag_5[(*input_data++) - START_OFFSET_5_BIT];
        temp       = tmp_buf[n++];
        xor_result ^= temp;
        // this copes with having 2 end sentinels
        if (*(input_data - 1) == MAG_END_SENTINEL &&
            *input_data == MAG_END_SENTINEL) {
            *input_data = 0;
        }
    }

    // if last character is not end sentinel
    // then make it so if App sets encoding is disabled
    if (bAddEndSentinel) {
        if (*(input_data - 1) != MAG_END_SENTINEL) {
            tmp_buf[n] = mag_5[(MAG_END_SENTINEL - START_OFFSET_5_BIT)];
            temp       = tmp_buf[n++];
            xor_result ^= temp;
        }
    }
    temp = xor_result;
    for (bit = 0; bit < 4; bit++) {
        if (xor_result & 0x80) {
            parity = ~parity;
        }
        xor_result <<= 1;
    }

    temp &= 0xf7;
    temp |= (parity & 0x08);
    tmp_buf[n++] = temp;
    tmp_buf[n]   = 0;

    return pack_bits(tmp_buf, n, 5, output_data, lead, 0);
}

/******************************************************************************

                       mag_encoder_setup_7bit_data()

 This will take a pointer to a string of MAG data for 7 bpc and use a
 look up table to convert into the correct 7 bit with parity bit
 reversed bytes and place these into the output string.
 Onto the end of this is placed the LRC and a terminating NULL

 Input parameters: void

 Returns:          the number of bits encoded

******************************************************************************/
int32_t mag_encoder_setup_7bit_data(char *     input_data,
                                    uint8_t *  output_data,
                                    const int  lead,
                                    const bool bAddEndSentinel) {
    int     n          = 0;
    int     bit        = 0;
    char    xor_result = 0;
    char    temp       = 0;
    uint8_t parity     = 0xFF;

    uint8_t tmp_buf[MAG_BUFSIZE] = {0};
    TRACE_IN;

    // If first character is a 'space' then skip
    if (*input_data == ' ') {
        input_data++;
    }

    while (*input_data && n < MAX_7_BIT_CHARS) {
        tmp_buf[n] = mag_7[(*input_data++) - START_OFFSET_7_BIT];
        temp       = tmp_buf[n++];
        xor_result ^= temp;
        // copes with 2 end sentinels
        if ((*(input_data - 1) == MAG_END_SENTINEL) &&
            (*input_data == MAG_END_SENTINEL)) {
            *input_data = 0;
        }
    }

    // if last character is not end sentinel then make it so if App sets
    // encoding is disabled
    if (bAddEndSentinel) {
        if (*(input_data - 1) != MAG_END_SENTINEL) {
            tmp_buf[n] = mag_7[(MAG_END_SENTINEL - START_OFFSET_7_BIT)];
            temp       = tmp_buf[n++];
            xor_result ^= temp;
        }
    }
    temp = xor_result;
    for (bit = 0; bit < 6; bit++) {
        if (xor_result & 0x80) {
            parity = ~parity;
        }
        xor_result <<= 1;
    }
    temp &= 0xfd;
    temp |= (parity & 0x02);
    tmp_buf[n++] = temp;
    tmp_buf[n]   = 0;

    return pack_bits(tmp_buf, n, 7, output_data, lead, 0);
}

/******************************************************************************

                        mag_encoder_setup_5bit_look_up()

  This generates the table of 16 bytes:- 5 bits for encoding 4 data bits
  plus parity bit reversed. ie data LSB is in byte MSB

  Input parameters: void

  Returns:          void

******************************************************************************/
void mag_encoder_setup_5bit_look_up(void) {
    int     byte   = 0;
    int     bit    = 0;
    char    temp1  = 0;
    char    temp2  = 0;
    uint8_t parity = 0;

    for (byte = 0; byte < 16; byte++) {
        temp1  = (char)byte;
        temp2  = 0;
        parity = 0xFF;
        for (bit = 0; bit < 5; bit++) {
            temp2 = (temp1 & 0x01) << (7 - bit) | temp2;
            if (temp1 & 0x01) {
                parity = ~parity;
            }
            temp1 >>= 1;
            temp1 &= 0x7F;
        }
        temp2 |= (parity & 0x08);
        mag_5[byte] = temp2;
    }
}

/******************************************************************************

                        mag_encoder_setup_7bit_look_up()

  This generates the table of 80 bytes:- 7 bits for encoding 6 data bits
  plus parity bit reversed. ie data LSB is in byte MSB

  Input parameters: void

  Returns:          void

 *****************************************************************************/
void mag_encoder_setup_7bit_look_up(void) {
    int     byte   = 0;
    int     bit    = 0;
    char    temp1  = 0;
    char    temp2  = 0;
    uint8_t parity = 0;

    for (byte = 0; byte < 80; byte++) {
        temp1  = (char)byte;
        temp2  = 0;
        parity = 0xFF;
        for (bit = 0; bit < 7; bit++) {
            temp2 = (temp1 & 0x01) << (7 - bit) | temp2;
            if (temp1 & 0x01) {
                parity = ~parity;
            }
            temp1 >>= 1;
            temp1 &= 0x7F;
        }
        temp2 |= (parity & 0x02);
        mag_7[byte] = temp2;
    }
}

/******************************************************************************

                       mag_encoder_setup_8bit_data()

 This will take a pointer to a string of MAG data for 8 bpc to convert
 into the correct 8 bit with parity bit reversed bytes and place these
 into the output string.  Onto the end of this is placed the LRC and a
 terminating NULL.

******************************************************************************/
int32_t mag_encoder_setup_8bit_data(char *     input_data,
                                    uint8_t *  output_data,
                                    const int  lead,
                                    const bool bAddEndSentinel) {

    int n = 0;
    int i = 0;

    uint8_t xor_result           = 0xff;
    uint8_t cur_byte             = 0;
    uint8_t tmp_buf[MAG_BUFSIZE] = {0};

    TRACE_IN;
    // ensure we have a start sentinel
    if ((*input_data & 0x7f) != JIS2_SENTINEL) {

        TRACE("No start sentinal (found '%c' instead)", *input_data);

        // start sentinel + parity bit
        tmp_buf[n] = (JIS2_SENTINEL | 0x80);

        // keep running total for LRC
        xor_result ^= tmp_buf[n++];
    }

    // not sure if MAX_8_BIT_CHARS includes sentinels and LRC
    while (*input_data && n < MAX_8_BIT_CHARS) {
        tmp_buf[n] = (*input_data++ & 0x7f);

        // compute parity bit and modify output data
        cur_byte = ((tmp_buf[n] ^ (tmp_buf[n] >> 4)) & 0x0f);
        tmp_buf[n] |= ((0x34cb00LU >> cur_byte) & 0x80);

        TRACE("[%02x - %c] -> [%02x - %c]",
              *(input_data - 1),
              *(input_data - 1),
              tmp_buf[n],
              tmp_buf[n]);

        // keep running total for LRC
        xor_result ^= tmp_buf[n++];

        // copes with 2 end sentinels
        if (((*(input_data - 1) & 0x7f) == JIS2_SENTINEL) &&
            ((*input_data & 0x7f) == JIS2_SENTINEL)) {
            *input_data = 0;
        }
    }

    // if last character is not end sentinel then make it so if App sets
    // encoding is disabled
    if (bAddEndSentinel) {
        if ((*(input_data - 1) & 0x7f) != JIS2_SENTINEL) {
            // end sentinel + parity bit
            tmp_buf[n] = (JIS2_SENTINEL | 0x80);

            // keep running total for LRC
            xor_result ^= tmp_buf[n++];
        }
    }
    // compute LRC and terminate data stream
    tmp_buf[n++] = xor_result;

    // reverse the bit order of each byte
    for (i = 0; i < n; ++i) {
        tmp_buf[i] = ((tmp_buf[i] >> 1) & 0x55) | ((tmp_buf[i] & 0x55) << 1);
        tmp_buf[i] = ((tmp_buf[i] >> 2) & 0x33) | ((tmp_buf[i] & 0x33) << 2);
        tmp_buf[i] = ((tmp_buf[i] >> 4) & 0x0f) | ((tmp_buf[i] & 0x0f) << 4);
    }

    return pack_bits(tmp_buf, n, 8, output_data, lead, 0);
}

/******************************************************************************

                       mag_encoder_setup_4bit_data()

 This will take a pointer to a string of MAG data for 4 bpc.
 odd length input is truncated by 1
 sample input 02fe80

  Input parameters: ptr to MAG data
                    ptr to output buffer
  Returns:          the number of bits encoded

******************************************************************************/
int32_t mag_encoder_setup_4bit_data(char *input_data, uint8_t *output_data) {
    int      len          = 0;
    int      num_bits     = 0;
    int      bit          = 0;
    int      val          = 0;
    int      maxinputlen  = MAG_BUFSIZE;
    char     HexValStr[2] = {0};
    uint8_t *pSrc         = NULL;

    // If first character is a 'space' then skip
    if (*input_data == ' ') {
        input_data++;
    }

    // determine if the input string is hex
    if (strtok((char *)input_data, "0123456789ABCDEFabcdef") != NULL) {
        return 0;
    }

    // if the string is not of even length null out the
    // last invalid passed hex char
    len = strlen((char *)input_data);
    if (len % 2) {
        output_data[len - 1] = 0;
        len--;
    }

    // if we are now reduced to a zero length string return..
    if (len == 0) {
        return 0;
    }

    // Clamp the length to the max
    if (len > maxinputlen) {
        len = maxinputlen;
    }

    pSrc = output_data;
    for (val = 0; val < len; val += 2) {
        HexValStr[0] = input_data[val];
        HexValStr[1] = input_data[val + 1];
        *pSrc++      = (uint8_t)strtoul((char *)&HexValStr[0], NULL, 16);
    }

    // now calculate the bit count
    // grab the last hex value
    val = output_data[(len >> 1) - 1];
    // c0..
    for (bit = 7; bit >= 0; bit--) {
        if (val & 1 << bit) {
            num_bits = 8 - bit;
        }
    }

    num_bits += ((len - 1) >> 1) * 8;

    // return the number of bits in the output data stream
    return num_bits;
}

/******************************************************************************

                       mag_encoder_setup_1bit_data()

 This will take a pointer to a string of MAG data for 1 bpc and pad
 if required.

  Input parameters: void

  Returns:          the number of bits encoded

******************************************************************************/
int32_t mag_encoder_setup_1bit_data(char *input_data, uint8_t *output_data) {

    int  n   = 0;
    int  len = 0;
    int  bit = 0;
    int  pad = 0;
    int  val = 0;
    char ch  = 0;

    char padded_data[MAG_BUFSIZE * 8] = {0};

    int      maxinputlen = MAG_BUFSIZE * 8;
    uint8_t *pSrc        = (uint8_t *)&padded_data;
    uint8_t *pSrcEnd     = NULL;

    // 10010111 00101111 0011     //input
    // 10010111 00101111 00110000 //pad out to 8
    //
    // 0x97, 0x2f, 0x30

    // If first character is a 'space' then skip
    if (*input_data == ' ') {
        input_data++;
    }

    len = strlen(input_data);

    if (len > maxinputlen) {
        len = maxinputlen;
    }

    mc_strncpy(padded_data, (char *)input_data, sizeof(padded_data));
    padded_data[len] = 0;

    // remove any trailing zeros
    pSrcEnd    = (uint8_t *)strrchr(padded_data, '1');
    *++pSrcEnd = 0;

    // redetermine length and required padding
    len = strlen((char *)padded_data);
    pad = 8 - (len % 8);
    if (pad == 8) {
        pad = 0;
    }

    if ((len + pad) > maxinputlen) {
        len = maxinputlen;
    }
    len += pad;

    for (val = 0; val < pad; val++) {
        strcat(padded_data, "0");
    }

    while (len && n < MAG_BUFSIZE) {
        val = 0;
        for (bit = 0; bit < 8; bit++) {
            ch = *pSrc++;
            if (ch != '0' && ch != '1') {
                // 0 or 1s only!
                return 0;
            }

            if (ch == '1') {
                val += 128 >> bit;
            }
            len--;
        }
        output_data[n++] = (uint8_t)val;
    }

    // return the number of bits in the output data stream
    return (n * 8) - pad;
}

