#include "rp2mag.h"
#include "comms.h"
#include "debug_utils.h"
#include "gndefs.h"
#include "magencd.h"
#include "magigen.h"
#include "rp2mag_encode.h"
#include "utils.h"
#include <stdbool.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define TRACK_KEYWORD_LEADER "TRACK"

/*****************************************************************************/
/*                   LOCAL FUNCTION PROTOTYPES                               */
/*****************************************************************************/
#include "rp2mag_local.h"

/**
 * @brief Pre-calculate the encoding for both 5bit and 7bit encoding, allowing
 *        quick encoding of a ascii character to an encoded character.
 */
STATIC void rp2mag_setup() {
    TRACE_IN;
    mag_encoder_setup_5bit_look_up();
    mag_encoder_setup_7bit_look_up();
    TRACE_OUT;
}

/**
 * @brief Output the relevant settings for the given mag track as xml,
 * appropriate for the Rio Pro 2 File Format
 * (http://ultrasvn2.magicard.local/~ctabmblyn/rp2fileformat/print-jobs/header/mag-options/track-options.html).
 *
 * @param output[] Array into which to write the xml for this track.
 * @param track    Pointer to a Rp2MagTrack to turn into xml.
 *
 * @return The number of bytes written to 'output'
 */
STATIC size_t track_to_xml(char output[], const struct Rp2MagTrack *track) {
    size_t cursor = 0;

    if (track->track_number != 0) {
        cursor += sprintf(&output[cursor],
                          "<track-options track=\"%d\">",
                          track->track_number);
        cursor +=
                sprintf(&output[cursor], "<bpi>%d</bpi>", track->settings.bpi);
        cursor += sprintf(&output[cursor],
                          "<num-bits>%zu</num-bits>",
                          track->data.encoded_len_bits);
        cursor += sprintf(&output[cursor], "</track-options>");
    }

    return cursor;
}

/**
 * @brief Compare two Rp2MagTrack structs.
 *        This will sort tracks by track_number, numerically, with the exception
 *        of track number 0 which is treated as infinetly large.
 *
 * @param lhs Left Rp2MagTrack struct.
 * @param rhs Right Rp2MagTrack struct.
 *
 * @return  Negative number if lhs is less than rhs,
 *          Positive number if rhs is less than lhs,
 *          0 if both are equal.
 */
STATIC int rp2mag_compare_tracks_by_track_number(const void *lhs,
                                                 const void *rhs) {
    if (((struct Rp2MagTrack *)lhs)->track_number == 0) {
        /* all tracks number 0 are unused, and so should go to the end */
        return 1;
    } else if (((struct Rp2MagTrack *)rhs)->track_number == 0) {
        return -1;
    } else {
        return ((struct Rp2MagTrack *)lhs)->track_number -
               ((struct Rp2MagTrack *)rhs)->track_number;
    }
}

/**
 * @brief Set all settings for the given Rp2Mag struct to 'sensible' values
 * appropriate for ISO encoding. Sets the Coercivity, Standard, Bits per
 * character, Bits per inch.
 *
 * @param mag Pointer to Rp2Mag struct which should be set to ISO defaults.
 */
STATIC void rp2mag_set_defaults(struct Rp2Mag *mag) {

    mag->settings.coercivity = HICO;
    mag->settings.standard   = ISO;
    for (size_t i = 0; i < RP2_MAG_NUM_TRACKS; i++) {
        if (i == 0) {
            mag->tracks[i].settings.bpc = BPC7;
        } else {
            mag->tracks[i].settings.bpc = BPC5;
        }

        if (i == 1) {
            mag->tracks[i].settings.bpi = BPI75;
        } else {
            mag->tracks[i].settings.bpi = BPI210;
        }
    }
}

/**
 * @brief Read the raw, user provided magnectic encoding string and extract any
 * settings provided within it, e.g. "~1,COEM,BPI75,%TRACKDATA?" will result in
 * the Coercivity being set to medium and the bits per inch being set to 75.
 *
 * @param mag Pointer to Rp2Mag struct containing between 1 and 3 Rp2MagTrack
 * structs to be processed.
 */
STATIC void rp2mag_raw_track_to_settings(struct Rp2Mag *mag) {

    for (size_t i = 0; i < RP2_MAG_NUM_TRACKS; i++) {
        /* strtok modifies the string in place, so take a copy to avoid changing
         * the original. */
        char *raw_mag_string = strdup(mag->tracks[i].data.raw);
        char *tok            = strtok(raw_mag_string, ",");

        if (tok != NULL) {
            /* first token must be "~N", where N is the track number. */
            if (tok[0] != '~') {
                ERR_STR("Track data *MUST* start with a tilde and the "
                        "track "
                        "number, "
                        "e.g. \"~1\"");
                /* Abort or continue with any other valid tracks? */
                free(raw_mag_string);
                continue;
            } else {
                mag->tracks[i].track_number = atoi(&tok[1]);
                TRACE("Setting track number to %d",
                      mag->tracks[i].track_number);
            }
        } else {
            free(raw_mag_string);
            continue;
        }

        /* Take the next token */
        tok = strtok(NULL, ",");

        /* Now lets see what other settings there are */
        while (tok != NULL) {
            if (strcmp("BPI75", tok) == 0) {
                mag->tracks[i].settings.bpi = BPI75;
            } else if (strcmp("BPI210", tok) == 0) {
                mag->tracks[i].settings.bpi = BPI210;
            } else if (strncmp("MPC", tok, strlen("MPC")) == 0) {
                TRACE("Found MPC of %s", tok);
                switch (atoi(tok + strlen("MPC"))) {
                    case 1:
                        mag->tracks[i].settings.bpc = BPC1;
                        break;
                    case 4:
                        mag->tracks[i].settings.bpc = BPC4;
                        break;
                    case 5:
                        mag->tracks[i].settings.bpc = BPC5;
                        break;
                    case 7:
                        mag->tracks[i].settings.bpc = BPC7;
                        break;
                    case 8:
                        mag->tracks[i].settings.bpc = BPC8;
                        break;
                }
            } else if (strncmp("COE", tok, strlen("COE")) == 0) {
                /* Coercivity applies to the whole mag stripe, not the
                 * individual tracks. */
                switch (*(tok + strlen("COE"))) {
                    case 'H':
                        mag->settings.coercivity = HICO;
                        break;
                    case 'M':
                        mag->settings.coercivity = MIDCO;
                        break;
                    case 'L':
                        mag->settings.coercivity = LOWCO;
                        break;
                }
            } else if (strncmp("MGV", tok, strlen("MGV")) == 0) {
                TRACE("Verify found. But which type? (%s)",
                      tok + strlen("MGV"));
                /* Verify */
                if (strncmp((tok + strlen("MGV")), "OFF", strlen("OFF")) == 0) {
                    TRACE_STR("verify off.");
                    mag->settings.verify = false;
                } else if (strncmp((tok + strlen("MGV")), "ON", strlen("ON")) ==
                           0) {
                    TRACE_STR("verify on.");
                    mag->settings.verify = true;
                }
            } else if (strncmp("JIS", tok, strlen("JIS")) == 0) {
                /* JIS */
                mag->settings.standard      = JIS;
                mag->tracks[0].settings.bpc = BPC8;
            }

            /* get the next token */
            tok = strtok(NULL, ",");
        } /* End while */
        free(raw_mag_string);
    } /* Next track */

    /*
     * Sort the tracks so that no matter the order we've parsed them,
     * track[0] is always the lowest numbered track, and all unused tracks
     * (with track_number = 0) are at the end of the array
     */
    qsort(mag->tracks,
          RP2_MAG_NUM_TRACKS,
          sizeof(struct Rp2MagTrack),
          &rp2mag_compare_tracks_by_track_number);
}

/**
 * @brief Extract any Mag track data from the options string recived by the
 * filter. E.g. "CFOverCoat=0PrintNoOvercoat TRACK1:~1,BPI5,%TRACKDATA?=true"
 * will populate tracks[0]'s raw data field with "~1,BPI5,%TRACKDATA?".
 * Essentially it'll remove the "TRACKX:" prefix and the "=true" suffix for any
 * mag option found.
 *
 * @param tracks[]    Array of Rp2MagTrack structs to populate with raw data.
 * @param options_string String provided to the filter from the lpr options (-o
 * option).
 */
STATIC void
rp2mag_extract_raw_track_data_from_string(struct Rp2MagTrack tracks[],
                                          char *             options_string) {
    char * tokens[RP2_MAG_NUM_TRACKS] = {0};
    size_t i                          = 0;

    TRACE("Parsing [%s]", options_string);
    if (strlen(options_string) < 1) {
        /* no arguments */
        return;
    }

    /* Start tokenising */
    char *arg = strtok(options_string, " ");

    if (arg == NULL) {
        /* No arguments */
        return;
    }

    do {
        if (strncmp(arg, TRACK_KEYWORD_LEADER, strlen(TRACK_KEYWORD_LEADER)) ==
            0) {
            tokens[i] = arg;
            i++;
        }
    } while ((arg = strtok(NULL, " ")) != NULL && i < RP2_MAG_NUM_TRACKS);

    for (i = 0; i < RP2_MAG_NUM_TRACKS && tokens[i] != NULL; i++) {
        /* clean up the option by stripping out the leading "TRACKX:"
         * and the trailing "=true" */
        char * start = (tokens[i] + strlen("TRACKX:"));
        size_t len   = 0;

        /* When running under the cupsfilter command, =true is appended to the
         * option string. Remove this if it is there. Otherwise, when running
         * under cups and the lpr command, the option just ends with a space, so
         * strlen will do the job. */
        char *tail = strstr(tokens[i], "=true");
        TRACE("Start = %s, end = %s", start, tail);
        if (NULL != tail) {
            /* +1 to convert from an index to a count */
            len = tail - start + 1;
        } else {
            /* +1 to convert from an index to a count */
            len = strlen(start) + 1;
        }

        mc_strncpy(tracks[i].data.raw, start, len);
    }
}

/**
 * @brief Encode the raw data within the Rp2MagTrack structs of mag, according
 * to their settings.
 *
 * @param mag Pointer to Rp2Mag struct containing the tracks to encode.
 */
STATIC void rp2mag_encode_tracks(struct Rp2Mag *mag) {

    rp2mag_setup();

    for (size_t i = 0; i < RP2_MAG_NUM_TRACKS; i++) {
        if (mag->tracks[i].track_number != 0) {
            /* we have a track */
            char *start_sentinel = mag->tracks[i].track_number == 1 ? "%" : ";";

            mag->tracks[i].data.encoded_len_bits = 0;

            /* The mag_encoder_* functions use strtok, which will insert null
             * characters into the input string. In order to preserve our "raw"
             * string, we need to encode a copy of the data. */
            char *data_to_be_encoded =
                    strdup(strstr(mag->tracks[i].data.raw, start_sentinel));

            TRACE("Encoding [%s] into track %d",
                  data_to_be_encoded,
                  mag->tracks[i].track_number);

            switch (mag->tracks[i].settings.bpc) {
                case BPC1:
                    break;
                case BPC4:
                    TRACE_STR("Encoding into 4 bits");
                    mag->tracks[i].data.encoded_len_bits =
                            mag_encoder_setup_4bit_data(
                                    data_to_be_encoded,
                                    mag->tracks[i].data.encoded);
                    break;
                case BPC5:
                    TRACE_STR("Encoding into 5 bits");
                    mag->tracks[i].data.encoded_len_bits =
                            mag_encoder_setup_5bit_data(
                                    data_to_be_encoded,
                                    mag->tracks[i].data.encoded,
                                    0,
                                    false);
                    break;
                case BPC7:
                    TRACE_STR("Encoding into 7 bits");
                    mag->tracks[i].data.encoded_len_bits =
                            mag_encoder_setup_7bit_data(
                                    data_to_be_encoded,
                                    mag->tracks[i].data.encoded,
                                    0,
                                    false);
                    break;
                case BPC8:
                    TRACE_STR("Encoding into 8 bits");
                    mag->tracks[i].data.encoded_len_bits =
                            mag_encoder_setup_8bit_data(
                                    data_to_be_encoded,
                                    mag->tracks[i].data.encoded,
                                    0,
                                    true);
                    break;
            }

            TRACE("Encoded into %zu bits",
                  mag->tracks[i].data.encoded_len_bits);

            /* store the number _whole_ bytes needed for the number of
             * encoded bits.
             */
            mag->tracks[i].data.encoded_len_round_bytes +=
                    mag->tracks[i].data.encoded_len_bits % 8 ?
                            (mag->tracks[i].data.encoded_len_bits / 8) + 1 :
                            mag->tracks[i].data.encoded_len_bits / 8;

            free(data_to_be_encoded);
        } /* End if track number != 0 */
    }     /* End for */
}

/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/

/**
 * @brief Turn a option string into a Rp2Mag struct.
 *        If there is mag data in the options string, extract the setting and
 * the data from it for all tracks listed. Encode the data according to the
 * settings provided.
 *
 * @param rp2_mag Pointer to the Rp2Mag struct to populate.
 * @param options_string String of all options sent to the filter, which may or
 * may not contain mag data.
 */
void rp2mag_parse_mag_command_line(struct Rp2Mag *rp2_mag,
                                   char *         options_string) {

    rp2mag_set_defaults(rp2_mag);
    rp2mag_extract_raw_track_data_from_string(rp2_mag->tracks, options_string);
    rp2mag_raw_track_to_settings(rp2_mag);
    rp2mag_encode_tracks(rp2_mag);
    return;
}

/**
 * @brief Report the sum of round bytes each track takes up. This is useful for
 * the <binary-size> tag of the Rp2 Fle Format.
 *
 * @param mag Pointer to the Rp2Mag to use.
 *
 * @return The sum of the number of bytes each track takes up, rounded up to the
 * nearest byte.
 */
size_t rp2mag_total_encoded_as_bytes(const struct Rp2Mag *mag) {
    size_t total = 0;
    for (size_t i = 0; i < RP2_MAG_NUM_TRACKS; i++) {
        if (mag->tracks[i].track_number > 0) {
            total += mag->tracks[i].data.encoded_len_round_bytes;
        }
    }
    return total;
}

/**
 * @brief For a given pointer to an Rp2Mag struct, generate appropriate xml for
 * use in a Rp2 PRN header. See
 * http://ultrasvn2.magicard.local/~ctamblyn/rp2fileformat/print-jobs-header/mag-options.html
 * for details.
 *
 * @param mag  Pointer to a Rp2Mag struct to generate the XML for.
 * @param output[]  Char array to write the mag header XML to.
 */
void rp2mag_to_xml(const struct Rp2Mag *mag, char output[]) {
    size_t cursor = 0;

    /* if there is no mag data, don't output anything */
    if (mag->tracks[0].data.encoded_len_bits < 1) {
        return;
    }

    cursor += sprintf(&output[cursor], "<mag-options>");

    /* Direction of encoding */
    cursor += sprintf(&output[cursor],
                      "<direction>%s</direction>",
                      mag->settings.standard == JIS ? "backwards" : "forwards");

    /* Verification */
    cursor += sprintf(&output[cursor],
                      "<verify-attempts>%d</verify-attempts>",
                      mag->settings.verify ? 3 : 0);
    /* Coercivity */
    cursor += sprintf(&output[cursor], "<coercivity>");
    switch (mag->settings.coercivity) {
        case HICO:
            cursor += sprintf(&output[cursor], "high");
            break;
        case MIDCO:
            cursor += sprintf(&output[cursor], "mid");
            break;
        case LOWCO:
            cursor += sprintf(&output[cursor], "low");
            break;
    }
    cursor += sprintf(&output[cursor], "</coercivity>");

    /* Now encode the tracks */
    for (size_t i = 0; i < RP2_MAG_NUM_TRACKS; i++) {

        cursor += track_to_xml(&output[cursor], &mag->tracks[i]);
    }

    cursor += sprintf(&output[cursor], "</mag-options>");
    output[cursor] = '\0';
}

