/*
 * Magicard Ltd
 *
 * CUPS Filter
 *
 * [ Linux ]
 * compile cmd: gcc -Wl,-rpath,/usr/lib -Wall -fPIC -O2 -o rastertoultra
 * rastertoultra.c magencd.c -lcupsimage -lcups
 * compile requires cups-devel-1.1.19-13.i386.rpm (version not neccessarily
 * important?)
 * find cups-devel location here:
 * http://rpmfind.net/linux/rpm2html/search.php?query=cups-devel&submit=Search+...&system=&arch=
 *
 * [ Mac OS X ]
 * compile cmd: gcc -Wall -fPIC -o rastertoultra rastertoultra.c magencd.c
 * -lcupsimage -lcups -arch i386 -DMACOSX
 * compile requires cupsddk-1.2.3.dmg (version not neccessarily important?)
 * find cupsddk location here: http://www.cups.org/software.php
 */

/*
 * Copyright (C) 2019 Magicard Ltd
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/*
 * Include necessary headers...
 */

#include <fcntl.h>
#include <math.h>
#include <cups/sidechannel.h>

#include "cmddefs.h"

#include "colour-profiles/included-colour-profiles.h"

#include "colortab.h"
#include "colrmtch.h"
#include "command.h"
#include "commandmb1.h"
#include "command_helper.h"
#include "comms.h"
#include "gndefs.h"
#include "magencd.h"
#include "magigen.h"
#include "utils.h"

// Set the version of the driver here.
#define DRV_VERSION "1.4.0"

#define MIN(a, b) (((a) < (b)) ? (a) : (b))
#define MAX(a, b) (((a) > (b)) ? (a) : (b))

#define RgbToGray(r, g, b) (((r)*74L + (g)*155L + (b)*27L) >> 8)

#define GetCValue(cmyk) ((uint8_t)((cmyk) >> 24))
#define GetMValue(cmyk) ((uint8_t)((cmyk) >> 16))
#define GetYValue(cmyk) ((uint8_t)((cmyk) >> 8))
#define GetKValue(cmyk) ((uint8_t)(cmyk))

#define RGB(r, g, b)                                                           \
    ((COLORREF)(((uint8_t)(r) | ((uint16_t)((uint8_t)(g)) << 8)) |             \
                (((uint32_t)(uint8_t)(b)) << 16)))
#define CMYK(c, m, y, k)                                                       \
    ((uint32_t)(((uint8_t)(c) << 24 | ((uint8_t)(m) << 16)) |                  \
                ((uint8_t)(y) << 8)) |                                         \
     ((uint8_t)(k)))

#define FOCUS_LEFT 0
#define FOCUS_CENTER 1
#define FOCUS_RIGHT 2

#define DATACANCEL_NO_USE 0
#define DATACANCEL_DOC 1

#define LOBYTE(w) ((uint8_t)(w))
#define HIBYTE(w) ((uint8_t)(((uint16_t)(w) >> 8) & 0xFF))
#define LOWORD(l) ((uint16_t)(l))
#define HIWORD(l) ((uint16_t)(((uint32_t)(l) >> 16) & 0xFFFF))
#define MAKELONG(a, b)                                                         \
    ((int32_t)(((uint16_t)(((uint32_t)(a)) & 0xffff)) |                        \
               ((uint32_t)((uint16_t)(((uint32_t)(b)) & 0xffff))) << 16))

#define MAKEWORD(a, b) ((b & 0xFF) | ((a & 0xff) << 8))
#define GetRValue(x) (LOBYTE(x))
#define GetGValue(x) (LOBYTE(((uint16_t)(x)) >> 8))
#define GetBValue(x) (LOBYTE((x) >> 16))
#define HALO_WIDTH (HALO_RADIUS * 2 + 1)
#define MAX_HALO_RADIUS 50
#define MAX_HALO_WIDTH (MAX_HALO_RADIUS * 2 + 1)
#define min(a, b) ((a) < (b) ? (a) : (b))

// macro converts RGB888 to RGB565
#define RGB_565(x)                                                             \
    ((((unsigned)(GetRValue(x)) << 8) & 0xF800) |                              \
     (((unsigned)(GetGValue(x)) << 3) & 0x7E0) |                               \
     (((unsigned)(GetBValue(x)) >> 3)))

// PDEVDATA flag definitions

// 1 = page level save in effect
#define PDEV_PGSAVED 0x00000001

// 1 = using color
#define PDEV_PRINTCOLOR 0x00000002

// 1 = Escape(STARTDOC) called
#define PDEV_STARTDOC 0x00000004

// 1 = EngWrite failed
#define PDEV_CANCELDOC 0x00000008

// 1 = doing image mask now
#define PDEV_DOIMAGEMASK 0x00000010

// 1 = not src first
#define PDEV_NOTSRCBLT 0x00000020

// 1 = using manual feed
#define PDEV_MANUALFEED 0x00000100

// 1 = Utils Procset sent
#define PDEV_UTILSSENT 0x00000200

// 1 = Pattern Bmp Procset sent
#define PDEV_BMPPATSENT 0x00000400

// 1 = Image Procset sent
#define PDEV_IMAGESENT 0x00000800

// 1 = procset part of header sent
#define PDEV_PROCSET 0x00004000

// 1 = withing save/restore of page
#define PDEV_WITHINPAGE 0x00008000

// 1 = this escape called
#define PDEV_EPSPRINTING_ESCAPE 0x00010000

// 1 = prefix TT font name with MSTT
#define PDEV_ADDMSTT 0x00020000

// 1 = don't want 1st page save/restore.
#define PDEV_NOFIRSTSAVE 0x00040000

// 1 = rawdata sent before procset sent.
#define PDEV_RAWBEFOREPROCSET 0x00080000

// set following a ResetPDEV, cleared at StartDoc
#define PDEV_RESETPDEV 0x00100000

// set to ignore GDI calls, cleared at StartPage
#define PDEV_IGNORE_GDI 0x00200000

// set to ingore DrvStartPage
#define PDEV_IGNORE_STARTPAGE 0x00400000

// same form/tray after DrvResetPDEV
#define PDEV_SAME_FORMTRAY 0x00800000

// driver signature
#define DRIVER_SIGNATURE 'MAGD'

typedef struct _RGB {
    uint8_t R;
    uint8_t G;
    uint8_t B;
} RGB;

typedef struct _charPixel {

    unsigned char r;
    unsigned char g;
    unsigned char b;
    unsigned char c;
    unsigned char m;
    unsigned char y;
} CHARPIXEL;

#define CHUNK 16384

// Idealy a power of 2,the driver assumes 4 though 8 works just fine
unsigned int GridDelta = 4;

#define GridSize (unsigned char)(256 / GridDelta) + 1
//
// used when defining the LUT's
#define MaxGrid 65

// LUT Processing
#define CYAN 0
#define MAGENTA 1
#define YELLOW 2

/*----------------------------------------------------------------------------*/

enum intents {
    PERCEPTUAL = 0,
    SATURATED,
    RELATIVE_COLORIMETRIC,
    ABSOLUTE_COLORIMETRIC,
    INTERNAL,
    NO_OF_INTENTS
};

enum LookupTable { RIO_LUT = 0, ENDURO_LUT, HELIX_LUT, PRO360_LUT };

struct command {
    int   length;
    char *command;
};

// local prototypes
static void initializeSettings(char *commandLineOptionSettings,
                                      struct settings_ *settings,
                                      PDEVDATA          pdev);

static ppd_choice_t *getOptionChoice(const char *choiceName, ppd_file_t *ppd);
static int getOptionChoiceIndex(const char *choiceName, ppd_file_t *ppd);
static const char *getOptionChoiceText(const char *choiceName, ppd_file_t *ppd);
static inline void getPageWidthPageHeight(ppd_file_t *      ppd,
                                          struct settings_ *settings);
static inline void myWrite(char *chData, uint32_t dwSize);

static uint32_t CMYK_to_RGB(uint8_t, uint8_t, uint8_t, uint8_t);

static void
CopyKPlaneToBuffer(PDEVDATA pdev,       // Pointer to our PDEVICE
                   uint8_t *lpSrc,      // Pointer to the image data. If this is
                                        // null, use pStripObj.
                   uint8_t * pSurface,  // Pointer to the surface object
                   PSETTINGS pSettings, // Pointer to the surface object
                   int32_t   lYOffset   // Current Y position
);

static unsigned char Cyan[MaxGrid][MaxGrid][MaxGrid][NO_OF_INTENTS];
static unsigned char Magenta[MaxGrid][MaxGrid][MaxGrid][NO_OF_INTENTS];
static unsigned char Yellow[MaxGrid][MaxGrid][MaxGrid][NO_OF_INTENTS];

unsigned int CMY[5];
unsigned int CMYX[3];

bool bLUTLoaded = false;

/*****************************************************************************
 *  myWrite()
 *      Output to file or spooler, binary data
 *
 *  Returns:
 *      None
 *****************************************************************************/
void myWrite(char *chData, uint32_t dwSize) {
    uint32_t i     = 0;
    char *   pByte = chData;

    for (; i < dwSize; i++) {
        putchar(*pByte++);
    }
}


inline void outputAsciiEncodedLength(int length) { printf("%d", length); }

inline void outputNullTerminator() { putchar(0x00); }

static ppd_choice_t *getOptionChoice(const char *choiceName, ppd_file_t *ppd) {
    ppd_choice_t *choice = NULL;
    ppd_option_t *option = NULL;

    // TRACE_IN;
    // TRACE("Looking for choice %s in ppd %s", choiceName, ppd->modelname);

   choice = PPDFINDMARKEDCHOICE(ppd, choiceName);
   if (choice == NULL) {
        // TRACE_STR("Marked choice is NULL. Looking for parent Option.");
      if ((option = PPDFINDOPTION(ppd, choiceName)) != NULL) {
            // TRACE("Option %s found.", option->text);
            // TRACE("Looking for the default choice (%s) for %s",
            // option->defchoice,
            // option->keyword);
         if ((choice = PPDFINDCHOICE(option, option->defchoice)) == NULL) {
                // TRACE_STR("No default choice found!");
         }
      } else {
            // TRACE("No option for %s", choiceName);
      }
   }

   if (choice) {
        // TRACE("Returing choice \"%s\"", choice->choice);
   } else {
        // TRACE("No choices for %s found. Returning NULL.", choiceName);
   }

    // TRACE_OUT;
   return choice;
}

static int getOptionChoiceIndex(const char *choiceName, ppd_file_t *ppd) {
    ppd_choice_t *choice = NULL;
   int ret = -1;

    // TRACE_IN;
   choice = getOptionChoice(choiceName, ppd);

   if (choice != NULL) {
      ret = atoi(choice->choice);
        // TRACE("Choice index is %d", ret);
   }

    // TRACE_OUT;
   return ret;
}

static const char *getOptionChoiceText(const char *choiceName,
                                       ppd_file_t *ppd) {
    const ppd_choice_t *choice = NULL;
    const char *        text   = NULL;
   TRACE_IN;

      choice = getOptionChoice(choiceName, ppd);

      if (choice != NULL) {
         text = choice->text;
         TRACE("Choice is \"%s\"", text);
      }

   TRACE_OUT;
   return text;
}

inline void getPageWidthPageHeight(ppd_file_t *      ppd,
                                   struct settings_ *settings) {
    ppd_choice_t *choice;
    ppd_option_t *option;

    char width[20];
    int  widthIdx;

    char height[20];
    int  heightIdx;

    char *pageSize;
    int   idx;

    int state;

    choice = PPDFINDMARKEDCHOICE(ppd, "PageSize");
    if (choice == NULL) {
        option = PPDFINDOPTION(ppd, "PageSize");
        choice = PPDFINDCHOICE(option, option->defchoice);
    }

    widthIdx = 0;
    memset(width, 0x00, sizeof(width));

    heightIdx = 0;
    memset(height, 0x00, sizeof(height));

    pageSize = choice->choice;
    idx      = 0;

    state = 0; // 0 = init, 1 = width, 2 = height, 3 = complete, 4 = fail

    while (pageSize[idx] != 0x00) {
        if (state == 0) {
            if (pageSize[idx] == 'X') {
                state = 1;

                idx++;
                continue;
            }
        } else if (state == 1) {
            if ((pageSize[idx] >= '0') && (pageSize[idx] <= '9')) {
                width[widthIdx++] = pageSize[idx];

                idx++;
                continue;
            } else if (pageSize[idx] == 'D') {
                width[widthIdx++] = '.';

                idx++;
                continue;
            } else if (pageSize[idx] == 'M') {
                idx++;
                continue;
            } else if (pageSize[idx] == 'Y') {
                state = 2;

                idx++;
                continue;
            }
        } else if (state == 2) {
            if ((pageSize[idx] >= '0') && (pageSize[idx] <= '9')) {
                height[heightIdx++] = pageSize[idx];

                idx++;
                continue;
            } else if (pageSize[idx] == 'D') {
                height[heightIdx++] = '.';

                idx++;
                continue;
            } else if (pageSize[idx] == 'M') {
                state = 3;
                break;
            }
        }

        state = 4;
        break;
    }

    if (state == 3) {
        settings->pageWidth  = atof(width);
        settings->pageHeight = atof(height);
    } else {
        settings->pageWidth  = 0;
        settings->pageHeight = 0;
    }
}

bool SetDefaultSettings(PSETTINGS pSettings) {
    pSettings->TargetPrinter = 0;

    TRACE_IN;
    TRACE("pSettings->Printer = [%d]", pSettings->Printer);

    // set a default..
    pSettings->OEM = OEM_ENDURO;

    if (pSettings->Printer) {
        switch (pSettings->Printer) {
                // 360 settings
            case PRO360:
            case IDM_SECURE_360:
            case ALPHACARD_PRO_700_360:
            case AUTHENTYS_PRO_360:
            case PHILOS_DTP_MK_3_360:
            case PLASCO_RIOPRO_LE_360:
            case SCREENCHECK_SC6500_360:
            case TITAN_T5_360:
            case PPC_ID3350E_360:
            case PRIDENTO_PRO_360:
            case PRO600:
            case CARDMAKER_360:
            case ALPHACARD_PRO_750_360:
            case PRO360NEO:
            case PRO300:
            case PRO500_D:
            case PRO500_K:
            case BOOD680:
            case FAGOOP360E:
            case AUTHENTYS300:
            case PRIDENTO300:
            case ALPHACARD_PRO_550:
            case IDMAKER_EDGE:
            case IDMAKER_APEX:
            case PPC_ID3000:
            case PPC_ID4000:
            case GOODCARDXR260D:
            case ING550:
            case GOODCARDXR360K:
            case DOH300:
            case MODEL_MAGICARD_PRONTO_100:
            case MODEL_MAGICARD_PRONTO_100_E:
            case MODEL_MAGICARD_100_NEO:
            case MODEL_IDVILLE_ID_MAKER_VALUE_MB1:
            case MODEL_MAGICARD_PRONTO_MB1:
            case MODEL_IDVILLE_ID_MAKER_ARC_MB1:
            case MODEL_PRICECARD_PRO_100:             
            case MODEL_DIGITAL_ID_EASYBADGE_3_0_MB1:
            case MODEL_MAGICARD_PRONTO_NEO_MB1:
            case MODEL_ALPHACARD_PRO_100_MB1:
            case MODEL_GOODCARD_XR160_MB1:
            case MODEL_ALPHACARD_PILOT_MB1:
            case MODEL_CENTENA_EDGE_MB1:
            case MODEL_MAGICARD_300_NEO_MB1:
            case MODEL_MAGICARD_600_NEO_MB1:
            case MODEL_MAGICARD_D_SUPPORTED_BY_BODNO_MB1: 
            case MODEL_GUDECARD_XR160PRO_MB1:
            case MODEL_BRADY_PA1100_MB1:
            case MODEL_BRADY_PA1300_MB1:{
                pSettings->OEM         = OEM_PRO360;
                pSettings->nPrintSpeed = UICBVAL_DefaultSpeed;
                pSettings->PaperHeight = 857;
                pSettings->nSharpness  = 0;
                break;
            }

            case PRO360_XTD:
            case PRIDENTO_PRO_360_XTD:
            case PRICECARD_PRO_FLEX_360:
            case MODEL_MAGICARD_PRICECARDPRO_FLEX_PLUS: 
            case MODEL_MAGICARD_E_PLUS_MB1:
            case MODEL_MAGICARD_E_PLUS_NEO_MB1:{
                TRACE_STR("Setting up defaults for FLEX+");
                pSettings->OEM         = OEM_PRO360;
                pSettings->nPrintSpeed = UICBVAL_DefaultSpeed;

                TRACE("pageType = %d", pSettings->pageType);
                if (pSettings->pageType == 2) {
                    // extd colour card
                   TRACE_STR("Setting defaults to 109mm colour");
                    pSettings->PaperHeight   = 1090;
                    pSettings->XXL_ImageType = UICBVAL_DoubleImage;
                } else if (pSettings->pageType == 3) {
                    // extd 140mm mono card
                   TRACE_STR("Setting defaults to 140mm monochrome");
                    pSettings->PaperHeight   = 1400;
                    pSettings->XXL_ImageType = UICBVAL_Extended_Monochrome;
                } else if (pSettings->pageType == 4) {
                    // extd 109mm mono card
                   TRACE_STR("Setting defaults to 109mm monochrome");
                    pSettings->PaperHeight   = 1090;
                    pSettings->XXL_ImageType = UICBVAL_Extended_Monochrome;
                } else if (pSettings->pageType == 5) {
                    // extd 128mm mono card
                   TRACE_STR("Setting defaults to 128mm monochrome");
                    pSettings->PaperHeight   = 1280;
                    pSettings->XXL_ImageType = UICBVAL_Extended_Monochrome;
                } else {
                   TRACE_STR("Setting defaults to single image (CR80)");
                    pSettings->XXL_ImageType =
                            UICBVAL_SingleImage; // riopro xtd only
                    pSettings->PaperHeight = 860;
                }
                break;
            }

            case HELIX:
            case PPC_RTP7500W:
            case AUTHENTYS_RETRAX:
            case SCREENCHECK_SC7500:
            case ULTIMA: {
                pSettings->OEM               = OEM_HELIX;
                pSettings->nPrintSpeed       = UICBVAL_DefaultSpeed;
                pSettings->nColourCorrection = 3;
                pSettings->PaperHeight       = 877;
                break;
            }

            case PRO:
            case IDM_SECURE:
            case AUTHENTYS_PRO:
            case PPC_ID3100:
            case PPC_ID3300:
            case RIO_PRO_XTENDED:
            case IDENTITYPRO:
            case PRIDENTOPRO:
            case PRIDENTOPROXTD:
            case DOPPIE_300:
            case SCREENCHECK_SC6500:
            case PLASCO_RIOPRO_LE:
            case JKE160C:
            case PHILOS_DTP_MK_3:
            case TITAN_T5: {
                TRACE("PRO settings->OEM  %u", pSettings->OEM);
                pSettings->OEM               = OEM_ENDURO;
                pSettings->nPrintSpeed       = UICBVAL_DefaultSpeed;
                pSettings->nColourCorrection = 3;
                if (pSettings->pageType == 2) {
                    // extd colour card
                    pSettings->PaperHeight   = 1080;
                    pSettings->XXL_ImageType = UICBVAL_DoubleImage;
                } else if (pSettings->pageType == 3) {
                    // extd mono card
                    pSettings->PaperHeight   = 1400;
                    pSettings->XXL_ImageType = UICBVAL_Extended_Monochrome;
                } else if (pSettings->pageType == 4) {
                    // extd mono card
                    pSettings->PaperHeight   = 1090;
                    pSettings->XXL_ImageType = UICBVAL_Extended_Monochrome;
                } else if (pSettings->pageType == 5) {
                    // extd mono card
                    pSettings->PaperHeight   = 1280;
                    pSettings->XXL_ImageType = UICBVAL_Extended_Monochrome;
                } else {
                    // riopro xtd only
                    pSettings->XXL_ImageType = UICBVAL_SingleImage;
                    pSettings->PaperHeight   = 860;
                }
                break;
            }
            case PRICECARD_PRO_FLEX: {
                pSettings->OEM               = OEM_ENDURO;
                pSettings->nPrintSpeed       = UICBVAL_DefaultSpeed;
                pSettings->nColourCorrection = 3;

                if (pSettings->pageType == 1) {
                    // extd mono card
                    pSettings->PaperHeight   = 1400;
                    pSettings->XXL_ImageType = UICBVAL_Extended_Monochrome;
                } else if (pSettings->pageType == 3) {
                    // extd mono card
                    pSettings->PaperHeight   = 1090;
                    pSettings->XXL_ImageType = UICBVAL_Extended_Monochrome;
                } else if (pSettings->pageType == 4) {
                    // extd mono card
                    pSettings->PaperHeight   = 1280;
                    pSettings->XXL_ImageType = UICBVAL_Extended_Monochrome;
                } else {
                    // riopro xtd only
                    pSettings->XXL_ImageType = UICBVAL_SingleImage;
                    pSettings->PaperHeight   = 860;
                }
                break;
            }
            case ALPHACARD_PRO_700: {
                pSettings->OEM               = OEM_ENDURO;
                pSettings->nPrintSpeed       = UICBVAL_DefaultSpeed;

                pSettings->nColourCorrection =
                        UICBVAL_ColourCorrection_RelColorimetric;
                pSettings->XXL_ImageType =
                        UICBVAL_SingleImage; // riopro xtd only
                pSettings->PaperHeight             = 860;

                break;
            }
            case ENDURO:
            case MC200:
            case IDMAKER:
            case IDM_ADVANTAGE:
            case AUTHENTYS:
            case P4500S:
            case PPC_ID2100:
            case PPC_ID2300:
            case FAGOO_P310E: {
                pSettings->OEM         = OEM_ENDURO;
                pSettings->nPrintSpeed = UICBVAL_DefaultSpeed;
                pSettings->nColourCorrection =
                        (pSettings->Printer == PPC_ID2300 ||
                         pSettings->Printer == FAGOO_P310E) ?
                                3 :
                                1;
                break;
            }
            case RIO_TANGO:
            case XXL: {
                pSettings->OEM                 = OEM_RIO;
                pSettings->TargetPrinter       = 1;
                pSettings->nColourCorrection   = 1;
                pSettings->nRollerTemperature  = 140;
                pSettings->nCardSpeed          = 4;
                pSettings->nPreLaminationDelay = 500;
                pSettings->nLaminationLength   = 86;
                pSettings->nStartOffset        = 0;
                pSettings->nEndOffset          = 16;
                pSettings->nFilmType           = 0;
                pSettings->XXL_ImageType       = UICBVAL_SingleImage;
                break;
            }
            case PRONTO:
            case IDM_VALUE:
            case P2500S:
            case PPC_ID2000:
            case PRIDENTO:
            case DOPPIE_100:
            case SCREENCHECK_SC2500:
            case XR160:
            case AUTHENTYS_PRONTO:
            case JKE700C:
            case PRIDENTO_OPEN_DATA:
            case ALPHACARD_PILOT:
            case PRICECARD_PRO_LITE:
            case PHILOS_DTP_MK_1:
            case PRONTO_NEO: {
                pSettings->OEM               = OEM_ENDURO;
                pSettings->nPrintSpeed       = UICBVAL_DefaultSpeed;
                pSettings->nColourCorrection = 1;
                break;
            }
            case ENDURO_PLUS:
            case AUTHENTYS_PLUS:
            case P4500S_PLUS:
            case IDM_ADVANT_PLUS:
            case PRIDENTO_PLUS:
            case ING171:
            case DOH:
            case ENDURO_3E:
            case PPC_ID2350E:
            case ILINKCARD_IT2600:
            case DOPPIE_200:
            case PCARD:
            case ORPHICARD:
            case SCREENCHECK_SC4500:
            case XR260:
            case PLASCO_ENDURO_LE:
            case JKE701C:
            case PRICECARD_PRO_DUO:
            case PHILOS_DTP_MK_2:
            case TITAN_T3:
            case ELLIADEN_CAROLINE:
            case CARDMAKER_260:
            case ENDURO_NEO:
            case NOVA: {
                pSettings->OEM               = OEM_ENDURO;
                pSettings->nPrintSpeed       = UICBVAL_DefaultSpeed;
                pSettings->nColourCorrection = 3;
                break;
            }
            case ALPHACARD_COMPASS:
            case ALPHACARD_PRO_500: {
                pSettings->OEM               = OEM_ENDURO;
                pSettings->nPrintSpeed       = UICBVAL_DefaultSpeed;
                pSettings->nColourCorrection = 4;
                break;
            }
            case PRIDENTO_PLUS_OPEN_DATA: {
                pSettings->OEM               = OEM_ENDURO;
                pSettings->nPrintSpeed       = UICBVAL_HighQuality;
                pSettings->nColourCorrection = 1;
                break;
            }
        }
    } else {
        TRACE("Enduro settings->OEM %u", pSettings->OEM);
        pSettings->Printer           = ENDURO;
        pSettings->OEM               = OEM_ENDURO;
        pSettings->nPrintSpeed       = UICBVAL_DefaultSpeed;
        pSettings->nColourCorrection = 1;
    }
    // to be populated by status monitor feedback
    pSettings->nPrintHeadPosition = 50;

    if (pSettings->Printer == P4500S_PLUS || pSettings->Printer == P2500S) {
        pSettings->CardBack.SecurityOptions.SecurityType =
                pSettings->CardFront.SecurityOptions.SecurityType = 4;
    }

    if (pSettings->Printer == IDENTITYPRO) {
        pSettings->CardBack.SecurityOptions.SecurityType =
                pSettings->CardFront.SecurityOptions.SecurityType = 1;
    }

    if (pSettings->Printer == IDENTITYPRO) {
        // default card front to K
        pSettings->CardFront.ColourFormat = UICBVAL_KResin;
    } else {
        // default card front to YMCK
        pSettings->CardFront.ColourFormat = UICBVAL_YMCK;
    }
    // default card back to K
    pSettings->CardBack.ColourFormat = UICBVAL_KResin;
    if (!HELIX_TYPE(pSettings) &&
        pSettings->Printer != PRICECARD_PRO_FLEX_360) {
        pSettings->CardFront.PrintOvercoat = true;
        pSettings->CardFront.HoloKote      = true;
    }
    // default on
    pSettings->CardFront.SecurityOptions.ColourHole = 1;
    // default to Area 6
    pSettings->CardFront.SecurityOptions.HoloKotePatch = 0x20;
    pSettings->CardBack.SecurityOptions.HoloKotePatch  = 0x20;
    pSettings->CardFront.SecurityOptions.HoloKoteMap   = 0xffffff;
    pSettings->CardBack.SecurityOptions.HoloKoteMap    = 0xffffff;

    // following polaroid model requested to default to YMC
    if (pSettings->Printer == P4500S_PLUS) {
        pSettings->CardFront.BlackOptions.AllBlackAs = 1;
        pSettings->CardBack.BlackOptions.AllBlackAs  = 1;
    }

    if (IDVILLE_TYPE(pSettings) || IDVILLE_360_TYPE(pSettings)) {
        pSettings->CardFront.BlackOptions.PicturesUseYMConly = 0;
        pSettings->CardBack.BlackOptions.PicturesUseYMConly  = 0;
    } else {
        pSettings->CardFront.BlackOptions.PicturesUseYMConly = 1;
        pSettings->CardBack.BlackOptions.PicturesUseYMConly  = 1;
    }

    pSettings->CardFront.BlackOptions.BlackTextUsesKResin    = 1;
    pSettings->CardFront.BlackOptions.MonoBitmapsUseKResin   = 1;
    pSettings->CardFront.BlackOptions.BlackPolygonsUseKResin = 1;
    pSettings->CardBack.BlackOptions.BlackTextUsesKResin     = 1;
    pSettings->CardBack.BlackOptions.MonoBitmapsUseKResin    = 1;
    pSettings->CardBack.BlackOptions.BlackPolygonsUseKResin  = 1;

    pSettings->AppDeterminesOrient = true;
    // DMORIENT_LANDSCAPE;
    pSettings->CardFront.CardOrient = pSettings->CardBack.CardOrient = 2;
    pSettings->bPauseSpooler                            = true;

    pSettings->nPrintHeadPower_YMC        = 50;
    pSettings->nPrintHeadPower_BlackResin = 50;
    pSettings->nPrintHeadPower_Overcoat   = 50;
    // request for magicard model default colour correction and ymc default
    // power 23/5/2017
    // 3e and plus and plasco models
    if (pSettings->Printer == ENDURO_PLUS || pSettings->Printer == ENDURO_3E ||
        pSettings->Printer == PLASCO_ENDURO_LE ||
        pSettings->Printer == PRICECARD_PRO_DUO ||
        pSettings->Printer == CARDMAKER_260 ||
        pSettings->Printer == ENDURO_NEO || pSettings->Printer == NOVA) {
        pSettings->nPrintHeadPower_YMC = 70;
        pSettings->nColourCorrection   = 4;
    }

    if (pSettings->Printer == PRO || pSettings->Printer == RIO_PRO_XTENDED ||
        pSettings->Printer == PLASCO_RIOPRO_LE) {

        pSettings->nPrintHeadPower_YMC = 60;
        pSettings->nColourCorrection   = 4;
    }

    pSettings->ErasePower_Start     = 50;
    pSettings->ErasePower_End       = 50;
    pSettings->WritePowerAdjustment = 50;

    pSettings->EraseArea_Left = 0; // model dependant?
    if (PRO360_TYPE(pSettings)) {
        pSettings->EraseArea_Width = 1013;
    } else {
        pSettings->EraseArea_Width = 1016;
    }

    pSettings->EraseArea_Bottom = 0;
    pSettings->EraseArea_Height = 642;

    pSettings->nBitsPerChar_Track2 = 1;
    pSettings->nBitsPerChar_Track3 = 1;
    pSettings->nBitsPerInch_Track2 = 1;

    if (pSettings->Printer == CARDMAKER_260 ||
        pSettings->Printer == CARDMAKER_360) {
        pSettings->CardFront.HoloKote  = false;
        pSettings->CardFront.Rotate180 = true;
    }

    return true;
}

void initializeSettings(char *            commandLineOptionSettings,
                        struct settings_ *settings,
                        PDEVDATA          pdev) {
    ppd_file_t *   ppd                  = NULL;
    cups_option_t *options              = NULL;
    char *         pOrientReq           = 0;
    char *         pCmdLineOptionTrack1 = NULL;
    char *         pCmdLineOptionTrack2 = NULL;
    char *         pCmdLineOptionTrack3 = NULL;
    char *         pCmdLineOptionEnd    = NULL;
    int            numOptions           = 0;
    char           buff[256];

    TRACE_STR("Opening PPD");
    ppd = PPDOPENFILE(getenv("PPD"));
    TRACE("PPD = %p", (void *)ppd);
    if (ppd == NULL) {
        ERR_STR("Unable to open PPD file. Exiting.");
        exit(1);
    }

    TRACE("cmdline = %s", commandLineOptionSettings);
    PPDMARKDEFAULTS(ppd);

    numOptions = CUPSPARSEOPTIONS(commandLineOptionSettings, 0, &options);
    if ((numOptions != 0) && (options != NULL)) {
        CUPSMARKOPTIONS(ppd, numOptions, options);

        CUPSFREEOPTIONS(numOptions, options);
    }
    memset(settings, 0x00, sizeof(struct settings_));
    // lpr -P<printername> -o"TRACK1:~1,%123123123?" -o"TRACK2:~2,2323232323?"
    // -o"TRACK3:~3,;2323232323#"
    // lpr -P<printername> -o"TRACK1:~1,BPI75,MPC5,%123123123?"
    // -o"TRACK2:~2,2323232323?" -o"TRACK3:~3,;2323232323#"

    // retrieve printer model
    settings->Printer = ppd->model_number;
    TRACE("settings->Printer = [%u]", settings->Printer);
    TRACE("ppd->model_number = [%u]", ppd->model_number);

    // selected page size
    // It turns out, this *must* be called before SetDefaultSettings.
    TRACE_STR("Getting selected PageSize from PPD...");
    settings->pageType = getOptionChoiceIndex("PageSize", ppd);
    TRACE("pageType here is %d", settings->pageType);

    // Setup defaults
    SetDefaultSettings(settings);
    TRACE("settings->OEM = %u", settings->OEM);

    if (settings->OEM == OEM_PRO360) {
        /* RP2 platform has its own Mag encoding */
        TRACE_STR("RP2 Mag Encoding initialising...");
        TRACE("CLI Options = [%s]", commandLineOptionSettings);
        rp2mag_parse_mag_command_line(&settings->rp2mag,
                                      commandLineOptionSettings);
    } else {
    InitializeMagneticEncodeTrackInfo(pdev, settings);

    pCmdLineOptionTrack1 = strstr(commandLineOptionSettings, "TRACK1");
    if (pCmdLineOptionTrack1) {
        pCmdLineOptionTrack1 += 7;
        pCmdLineOptionEnd = strstr(pCmdLineOptionTrack1, " ");
            if (pCmdLineOptionEnd == NULL) {
                // no trailing space - use the end of the string
                pCmdLineOptionEnd =
                        &pCmdLineOptionTrack1[strlen(pCmdLineOptionTrack1)];
            }
            memset(buff, 0x00, sizeof(buff));
            memcpy((uint8_t *)&buff,
                   pCmdLineOptionTrack1,
                   (int)(pCmdLineOptionEnd - pCmdLineOptionTrack1));
            TRACE("Passing to record %s", buff);
            RecordMagneticEncodeTrackInfo(pdev, buff, settings);
            TRACE_STR("FINISHED RECORD MAG TRACK1");
            TRACE("pdev->magTrackInfo[MAG_TRACK_INDEX1].TrackData = %s",
                  pdev->magTrackInfo[MAG_TRACK_INDEX1].TrackData);
            TRACE("pdev->magTrackInfo[MAG_TRACK_INDEX1].bComplete = %s",
                  STR_BOOLEAN(pdev->magTrackInfo[MAG_TRACK_INDEX1].bComplete));
        }

    pCmdLineOptionTrack2 = strstr(commandLineOptionSettings, "TRACK2");
    if (pCmdLineOptionTrack2) {
        pCmdLineOptionTrack2 += 7;
        pCmdLineOptionEnd = strstr(pCmdLineOptionTrack2, " ");
            if (pCmdLineOptionEnd == NULL) {
                // no trailing space - use the end of the string
                pCmdLineOptionEnd =
                        &pCmdLineOptionTrack2[strlen(pCmdLineOptionTrack2)];
            }
            memset(buff, 0x00, sizeof(buff));
            memcpy((uint8_t *)&buff,
                   pCmdLineOptionTrack2,
                   (int)(pCmdLineOptionEnd - pCmdLineOptionTrack2));
            TRACE("Passing to record %s", buff);
            RecordMagneticEncodeTrackInfo(pdev, buff, settings);
            TRACE_STR("FINISHED RECORD MAG TRACK2");
            TRACE("pdev->magTrackInfo[MAG_TRACK_INDEX2].TrackData = %s",
                  pdev->magTrackInfo[MAG_TRACK_INDEX2].TrackData);
            TRACE("pdev->magTrackInfo[MAG_TRACK_INDEX2].bComplete = %s",
                  STR_BOOLEAN(pdev->magTrackInfo[MAG_TRACK_INDEX2].bComplete));
        }

        TRACE("commandLineOptionSettings = %s", commandLineOptionSettings);
    pCmdLineOptionTrack3 = strstr(commandLineOptionSettings, "TRACK3");
        TRACE("strstr found %s", pCmdLineOptionTrack3);
    if (pCmdLineOptionTrack3) {
            TRACE_AT;
        pCmdLineOptionTrack3 += 7;
        pCmdLineOptionEnd = strstr(pCmdLineOptionTrack3, " ");
            if (pCmdLineOptionEnd == NULL) {
                // no trailing space - use the end of the string
                pCmdLineOptionEnd =
                        &pCmdLineOptionTrack3[strlen(pCmdLineOptionTrack3)];
            }
            TRACE_AT;
            memset(buff, 0x00, sizeof(buff));
            memcpy((uint8_t *)&buff,
                   pCmdLineOptionTrack3,
                   (int)(pCmdLineOptionEnd - pCmdLineOptionTrack3));
            TRACE("Passing to record %s", buff);
            RecordMagneticEncodeTrackInfo(pdev, buff, settings);
            TRACE_STR("FINISHED RECORD MAG TRACK3");
            TRACE("pdev->magTrackInfo[MAG_TRACK_INDEX3].TrackData = %s",
                  pdev->magTrackInfo[MAG_TRACK_INDEX3].TrackData);
            TRACE("pdev->magTrackInfo[MAG_TRACK_INDEX3].bComplete = %s",
                  STR_BOOLEAN(pdev->magTrackInfo[MAG_TRACK_INDEX3].bComplete));
        }
    }

    pOrientReq = strstr(commandLineOptionSettings, "orientation-requested");
    if (pOrientReq) {
        TRACE_STR("ORIENTATION REQUEST...");
        if (pOrientReq[22] == '4') {
            TRACE_STR("SETTING LANDSCAPE...");
            settings->orientation = 1;
        }
    }
    settings->orientation = 1;

    if (settings->Printer >= HELIX) {
        settings->bytesPerScanLine    = 3108; // 1036 * 3
        settings->bytesPerScanLineStd = 3108;
    } else {
        settings->bytesPerScanLine    = 1926; // 642 * 3
        settings->bytesPerScanLineStd = 1926;
    }

    if (settings->OEM == OEM_HELIX) {
        settings->ISO7810        = getOptionChoiceIndex("BendSolution", ppd);
        settings->CFHolokoteSlot = getOptionChoiceIndex("CFHolokoteSlot", ppd);
        settings->CBHolokoteSlot = getOptionChoiceIndex("CBHolokoteSlot", ppd);
        settings->nHLXPrintHeadPowerYMC =
                getOptionChoiceIndex("PrintheadPowerYMC", ppd);
        settings->nHLXPrintHeadPowerK =
                getOptionChoiceIndex("PrintheadPowerK", ppd);
        settings->nHLXPrintHeadPowerOvercoat =
                getOptionChoiceIndex("PrintheadPowerOvercoat", ppd);
    } else if (settings->OEM == OEM_PRO360) {
        settings->CFHolokoteSlot =
                getOptionChoiceIndex("CFHolokoteSlot", ppd) + 1;
        settings->CBHolokoteSlot =
                getOptionChoiceIndex("CBHolokoteSlot", ppd) + 1;
        settings->CardFront.SecurityOptions.HoloKote_XAdjust =
                getOptionChoiceIndex("CFHolokoteXAdjust", ppd) - 25;
        settings->CardFront.SecurityOptions.HoloKote_YAdjust =
                getOptionChoiceIndex("CFHolokoteYAdjust", ppd) - 25;
        settings->CardBack.SecurityOptions.HoloKote_XAdjust =
                getOptionChoiceIndex("CBHolokoteXAdjust", ppd) - 25;
        settings->CardBack.SecurityOptions.HoloKote_YAdjust =
                getOptionChoiceIndex("CBHolokoteYAdjust", ppd) - 25;

        // hard code to patch 6
        settings->CardFront.SecurityOptions.HoloPatchOffset.left   = 808;
        settings->CardFront.SecurityOptions.HoloPatchOffset.width  = 175;
        settings->CardFront.SecurityOptions.HoloPatchOffset.bottom = 447;
        settings->CardFront.SecurityOptions.HoloPatchOffset.height = 175;
        settings->CardFront.SecurityOptions.UsewithLaminate =
                getOptionChoiceIndex("CFUsewithLaminate", ppd);
        settings->CardBack.SecurityOptions.UsewithLaminate =
                getOptionChoiceIndex("CBUsewithLaminate", ppd);
        if (settings->Printer == PRO600) {
            settings->xdpi = getOptionChoiceIndex("Resolution", ppd);
        }
        TRACE("MODEL - 360! %u", settings->Printer);
    }

    settings->Duplex = getOptionChoiceIndex("UTDuplex", ppd);

    settings->CardFront.ColourFormat =
            getOptionChoiceIndex("CFColourFormat", ppd);

    settings->CardFront.BlackOptions.AllBlackAs =
            getOptionChoiceIndex("CFBlackResin", ppd);
    settings->CardFront.PrintOvercoat =
            getOptionChoiceIndex("CFOverCoat", ppd) == 1;
    settings->CardFront.HoloKote = getOptionChoiceIndex("CFHoloKote", ppd) == 1;

    settings->CardFront.HoloPatch =
            getOptionChoiceIndex("CFHoloPatch", ppd) == 1;

    if (settings->CardFront.HoloPatch &&
        (settings->Printer == ENDURO || settings->Printer == ENDURO_PLUS)) {
        settings->CardFront.SecurityOptions.ColourHole = 1;
    }

    settings->CardFront.CardOrient = getOptionChoiceIndex("Orientation", ppd);
    settings->CardFront.Rotate180 =
            getOptionChoiceIndex("CFRotation", ppd) == 1;
    settings->CardFront.OvercoatOptions.Holes =
            getOptionChoiceIndex("CFOvercoatHole", ppd);
    settings->CardFront.SecurityOptions.SecurityType =
            getOptionChoiceIndex("CFHolokoteType", ppd);

    settings->CardFront.SecurityOptions.Rotation =
            getOptionChoiceIndex("CFHolokoteRotation", ppd);

    settings->CardBack.ColourFormat =
            getOptionChoiceIndex("CBColourFormat", ppd);
    settings->CardBack.BlackOptions.AllBlackAs =
            getOptionChoiceIndex("CBBlackResin", ppd);
    settings->CardBack.PrintOvercoat =
            getOptionChoiceIndex("CBOverCoat", ppd) == 1;
    settings->CardBack.HoloKote = getOptionChoiceIndex("CBHoloKote", ppd) == 1;
    settings->CardBack.HoloPatch =
            getOptionChoiceIndex("CBHoloPatch", ppd) == 1;
    settings->CardBack.CardOrient = getOptionChoiceIndex("Orientation", ppd);
    settings->CardBack.Rotate180 = getOptionChoiceIndex("CBRotation", ppd) == 1;

    settings->CardBack.OvercoatOptions.Holes =
            getOptionChoiceIndex("CBOvercoatHole", ppd);
    settings->CardBack.SecurityOptions.SecurityType =
            getOptionChoiceIndex("CBHolokoteType", ppd);
    settings->CardBack.SecurityOptions.Rotation =
            getOptionChoiceIndex("CBHolokoteRotation", ppd);

    settings->nColourCorrection = getOptionChoiceIndex("ColourCorrection", ppd);

    TRACE("settings->nColourCorrection %d", settings->nColourCorrection);

    settings->nPrintSpeed = getOptionChoiceIndex("ResinQuality", ppd);
    settings->bColourSure = getOptionChoiceIndex("ColourSure", ppd) == 1;
    settings->nSharpness  = -2 + getOptionChoiceIndex("Sharpness", ppd);
    settings->bDisableStatusPolling =
            getOptionChoiceIndex("StatusPolling", ppd);

    settings->nPrintHeadPower_YMC =
            getOptionChoiceIndex("PrintheadPowerYMC", ppd);

    settings->nPrintHeadPower_BlackResin =
            getOptionChoiceIndex("PrintheadPowerK", ppd);
    settings->nPrintHeadPower_Overcoat =
            getOptionChoiceIndex("PrintheadPowerOvercoat", ppd);
    settings->nImagePosition_UpDown =
            -15 + getOptionChoiceIndex("PrintHeadPos", ppd);
    settings->nImagePosition_Start =
            -50 + getOptionChoiceIndex("ImagePosStart", ppd);
    settings->nImagePosition_End =
            -50 + getOptionChoiceIndex("ImagePosEnd", ppd);
    settings->bEraseBeforePrint =
            getOptionChoiceIndex("RewritablecardsEraseBeforePrint", ppd) == 0;

    if (getOptionChoiceIndex("ErasePowerStart", ppd) != -1) {
        settings->ErasePower_Start =
                getOptionChoiceIndex("ErasePowerStart", ppd);
    }
    settings->ErasePower_End   = getOptionChoiceIndex("ErasePowerEnd", ppd);
    settings->WritePowerAdjustment = getOptionChoiceIndex("WritePower", ppd);

    getPageWidthPageHeight(ppd, settings);

    /* Print Mode
     * High Speed/Normal in the ppd/UI, Normal/Enhanced in the printer
     * firmware
    */

    settings->highQualityResin = PRINT_MODE_NORMAL;
    {
        const char *choice = getOptionChoiceText("HighQualityResin", ppd);
        if (choice != NULL && strcmp(choice, "Yes") == 0) {
            settings->highQualityResin = PRINT_MODE_ENHANCED;
        }
    }

    settings->highQualityYMC = PRINT_MODE_NORMAL;
    {
        const char *choice = getOptionChoiceText("HighQualityYMC", ppd);
        if (choice != NULL && strcmp(choice, "Yes") == 0) {
            settings->highQualityYMC = PRINT_MODE_ENHANCED;
        }
    }

    PPDCLOSE(ppd);
    TRACE("MODEL - %u\n", settings->Printer);
}

/*****************************************************************************
 *  OutputComma()
 *      Send command for comma
 *
 *  Returns:
 *      None
 *****************************************************************************/
void OutputComma() {
    // Send command for comma
    printf(CMD_STR_COMMA);
}

/*****************************************************************************
 *  OutputFileSeparator()
 *      Send command for file separator
 *
 *  Returns:
 *      None
 *****************************************************************************/
void OutputFileSeparator() {
    // Send command for file separator
    printf(CMD_STR_FILE_SEPARATOR);
}

/*****************************************************************************
 *  OutputETXCommand()
 *      Send command for End of Transmission
 *
 *  Returns:
 *      None
 *****************************************************************************/
void OutputETXCommand() {
    // Send command for end-of-page
    printf(CMD_STR_ETX);
}

#ifdef BUMPVERSION
extern uint16_t ICCValue;
#endif

//##############################################################################



/*****************************************************************************
 *  WriteBitsPerChar()
 *      Outputs the number of Bits per char if not default (i.e. ISO)
 *      Alto/Opera/Tempo are always ISO
 *
 *  Returns:
 *      None
 *****************************************************************************/
void WriteBitsPerChar(struct settings_ *settings, uint32_t Track_no) {
    switch (Track_no) {
        case MAG_TRACK_INDEX1:
            switch (settings->nBitsPerChar_Track1) {
                // Default (ISO standard) is 7 bits - no header entry required
                case 1: // UICBVAL_BitsPerChar_Track1_5:
                    printf(CMD_STR_BITS_PER_CHAR, BITS_PER_CHAR_5);
                    break;
                case 2: // UICBVAL_BitsPerChar_Track1_1:
                    printf(CMD_STR_BITS_PER_CHAR, BITS_PER_CHAR_1);
                    break;
            }
            break;

        case MAG_TRACK_INDEX2:
            switch (settings->nBitsPerChar_Track2) {
                // Default (ISO standard) is 5 bits - no header entry required
                case 0: // UICBVAL_BitsPerChar_Track2_7:
                    printf(CMD_STR_BITS_PER_CHAR, BITS_PER_CHAR_7);
                    break;
                case 2: // UICBVAL_BitsPerChar_Track2_1:
                    printf(CMD_STR_BITS_PER_CHAR, BITS_PER_CHAR_1);
                    break;
            }
            break;

        case MAG_TRACK_INDEX3:
            switch (settings->nBitsPerChar_Track3) {
                // Default (ISO standard) is 5 bits - no header entry required
                case 0: // UICBVAL_BitsPerChar_Track3_7:
                    printf(CMD_STR_BITS_PER_CHAR, BITS_PER_CHAR_7);
                    break;
                case 2: // UICBVAL_BitsPerChar_Track3_1:
                    printf(CMD_STR_BITS_PER_CHAR, BITS_PER_CHAR_1);
                    break;
            }
            break;
    }
}

/*****************************************************************************
 *  WriteBitsPerInch()
 *      Outputs the number of Bits per Inch if not default (i.e. ISO)
 *      Alto/Opera/Tempo are always ISO
 *
 *  Returns:
 *      None
 *****************************************************************************/
void WriteBitsPerInch(struct settings_ *settings, uint32_t Track_no) {

    switch (Track_no) {
        case MAG_TRACK_INDEX1:
            // Default (ISO standard) is 210 bits per inch.
            if (settings->nBitsPerInch_Track1 == 1) {
                /*UICBVAL_BitsPerInch_Track1_75*/
                printf(CMD_STR_BITS_PER_INCH, BITS_PER_INCH_75);
            }
            break;

        case MAG_TRACK_INDEX2:
            // Default (ISO standard) is 75 bits per inch.
            if (settings->nBitsPerInch_Track2 == 0) {
                /*UICBVAL_BitsPerInch_Track2_210*/
                printf(CMD_STR_BITS_PER_INCH, BITS_PER_INCH_210);
            }
            break;

        case MAG_TRACK_INDEX3:
            // Default (ISO standard) is 210 bits per inch.
            if (settings->nBitsPerInch_Track3 == 1) {
                /*UICBVAL_BitsPerInch_Track3_75*/
                printf(CMD_STR_BITS_PER_INCH, BITS_PER_INCH_75);
            }
            break;
    }
}

/*****************************************************************************
 *  GetDataPresent()
 *      Check if we have seen any colour in any plane.
 *
 *  Parameters:
 *      pdev      Pointer to our PDEV
 *      bFront    true = Front page (or simplex) : false = Back page
 *
 *  Returns:
 *      Return a mask indicating whether data is in a plane
 *            Bit 0 = Yellow
 *            Bit 1 = Magenta
 *            Bit 2 = Cyan
 *            Bit 3 = Black
 *****************************************************************************/
uint8_t GetDataPresent(PDEVDATA pdev, bool bFront) {
    LPSPOOLMEMINFO lpSplMem   = 0;
    uint32_t       ulColour   = 0;
    uint8_t        ColourMask = 0;

    for (ulColour = 0; ulColour < 4; ulColour++) {
        if (bFront) {
            lpSplMem = &pdev->lpSplMemFront[ulColour];
        } else {
            lpSplMem = &pdev->lpSplMemBack[ulColour];
        }

        if ((lpSplMem->lpBuffer != 0) && lpSplMem->bDataInPlane &&
            (lpSplMem->ulDataSize != 0)) {
            // Set mask bit to show there is data in this plane
            ColourMask |= (1 << ulColour);
        }
    }
    return ColourMask;
}

/*****************************************************************************
 *  LaminatorCommand()
 *      Output laminator commands.
 *      includes the appropriate laminator command in the header.
 *      Dependent on the lamination settings and the print settings (front,
 *      back or both sides)
 *
 *  Returns:
 *      None
 *****************************************************************************/
void LaminatorCommand(PDEVDATA pdev, struct settings_ *settings) {
    int LamSetting   = 0;
    int EjectSetting = 1;
    int FilmType     = OVERLAY_FILM;

    if (settings->bLaminateOnly) {
        switch (settings->nLaminateSide) {
            case UICBVAL_Laminating_None:
                LamSetting   = 0;
                EjectSetting = 1;
                break;
            case UICBVAL_Laminating_Front:
                LamSetting   = 1;
                EjectSetting = 1;
                break;
            case UICBVAL_Laminating_Back:
                LamSetting   = 2;
                EjectSetting = 2;
                break;
            case UICBVAL_Laminating_Both:
                LamSetting   = 3;
                EjectSetting = 2;
                break;
        }
    } else {
        if ((settings->Duplex == UICBVAL_Duplex_BothSides) &&
            (GetDataPresent(pdev, false))) {
            // Duplex with data on both sides
            switch (settings->nLaminateSide) {
                case UICBVAL_Laminating_None:
                    LamSetting   = 8;
                    EjectSetting = 2;
                    break;
                case UICBVAL_Laminating_Front:
                    LamSetting   = 9;
                    EjectSetting = 1;
                    break;
                case UICBVAL_Laminating_Back:
                    LamSetting   = 10;
                    EjectSetting = 2;
                    break;
                case UICBVAL_Laminating_Both:
                    LamSetting   = 11;
                    EjectSetting = 1;
                    break;
            }
        } else {
            // Duplex_Front, Duplex_Back, or Duplex_BothSides with data on front
            // only
            switch (settings->nLaminateSide) {
                case UICBVAL_Laminating_None:
                    LamSetting   = 4;
                    EjectSetting = 1;
                    break;
                case UICBVAL_Laminating_Front:
                    LamSetting   = 5;
                    EjectSetting = 1;
                    break;
                case UICBVAL_Laminating_Back:
                    LamSetting   = 6;
                    EjectSetting = 2;
                    break;
                case UICBVAL_Laminating_Both:
                    LamSetting   = 7;
                    EjectSetting = 2;
                    break;
            }
        }
    }

    // Eject Side
    printf(CMD_STR_EJECTSIDE, EjectSetting);

    // LAM Command
    printf(CMD_STR_LAMINATOR_COMMAND, LamSetting);

    // Roller Temperature
    printf(CMD_STR_ROLLER_TEMP, settings->nRollerTemperature);

    // Card Speed
    printf(CMD_STR_CARD_SPEED, settings->nCardSpeed);

    // Pre-Lamination Delay
    printf(CMD_STR_PRELAM_DELAY, settings->nPreLaminationDelay);

    // Card Length
    printf(CMD_STR_CARD_LENGTH, settings->nLaminationLength);

    // Start Offset
    printf(CMD_STR_START_OFFSET, settings->nStartOffset);

    // End Offset
    printf(CMD_STR_END_OFFSET, settings->nEndOffset);

    // Film Type
    switch (settings->nFilmType) {
        case UICBVAL_LamFilm_Overlay:
            FilmType = OVERLAY_FILM;
            break;
        case UICBVAL_LamFilm_Patch:
            FilmType = PATCH_FILM;
            break;
    }
    printf(CMD_STR_FILM_TYPE, FilmType);
}

/*****************************************************************************
 *  RioProHKTValue()
 *      Determines the HKT Value to be sent to a RioPro (Type) printer
 *
 *  Returns:
 *      int value for the HKT command
 *****************************************************************************/
int RioProHKTValue(struct settings_ *settings, int UICB_Value) {
    // This code is dependent on the fact that the settings for the back are the
    // same as those for the front.  This is because we cannot have multiple
    // case labels with the same value
    if (IDMAKER_MODEL(settings) || IDM_VALUE_MODEL(settings) ||
        IDM_ADVANTAGE_MODEL(settings) || IDM_ADV_PLUS_MODEL(settings)) {
        switch (UICB_Value) {
            default:
            case 0:
                return 6; // IDVille Logo
            case 1:
                return 7; // Seal
            case 2:
                return 8; // Eagle
            case 3:
                return 9; // Globe
            case 4:
                return 5; // Flex
        }
    } else if (IDM_SECURE_MODEL(settings)) {
        switch (UICB_Value) {
            default:
            case 0:
                return 1; // IDVille Logo
            case 1:
                return 2; // Seal
            case 2:
                return 3; // Eagle
            case 3:
                return 4; // Globe
            case 4:
                return 5; // Flex
        }
    } else if (PRIDENTO_TYPE(settings)) {
        switch (UICB_Value) {
            default:
            case 0:
                return 254; // Pridento Holokote
            case 1:
                return 255; // Pridento Flex
            case 2:
                return 2; // Rings
            case 3:
                return 3; // Globe
            case 4:
                return 4; // Waves
        }
    } else if (AUTHENTYS_TYPE(settings)) {
        switch (UICB_Value) {
            default:
            case 0:
                return 1; // UltraSecure Key
            case 1:
                return 2; // Rings
            case 2:
                return 3; // Globe
            case 3:
                return 5; // Flex
            case 4:
                return 4; // Authentys A
        }
    } else if (IDENTITY_PRO_TYPE(settings)) {
        switch (UICB_Value) {
            default:
            case 0:
                return 2; // Rings
            case 1:
                return 3; // Globe
            case 2:
                return 4; // Cubes
            case 3:
                return 5; // Flex
        }
    } else if (ALPHACARD_PILOT_MODEL(settings) ||
        ALPHACARD_COMPASS_MODEL(settings) ||
        ALPHACARD_PRO_100_MODEL(settings) ||
        ALPHACARD_PRO_500_MODEL(settings)) {
        switch (UICB_Value) {
            default:
            case 0:
                return 5; // ALPHAGUARD_TILE
            case 1:
                return 2; // Rings
            case 2:
                return 3; // Globe
            case 3:
                return 4; // Wavelength
        }
    } else if (ALPHACARD_PRO_700_MODEL(settings)) {
        switch (UICB_Value) {
            default:
            case 0:
                return 1; // Flex
            case 1:
                return 2; // Rings
            case 2:
                return 3; // Globe
            case 3:
                return 4; // Cubes
            case 4:
                return 5; // Flex
        }
    } else if (PHILOS_TYPE(settings)) {
        switch (UICB_Value) {
            default:
            case 0:
                return 1; // Flex
            case 1:
                return 2; // Rings
            case 2:
                return 3; // Globe
            case 3:
                return 4; // Wavelength
        }
    } else if (TITAN_T3_MODEL(settings)) {
        switch (UICB_Value) {
            default:
            case 0:
                return 1; // Flex
            case 1:
                return 2; // Rings
            case 2:
                return 3; // Globe
            case 3:
                return 4; // Wavelength
        }
    } else if (TITAN_T5_MODEL(settings)) {
        switch (UICB_Value) {
            default:
            case 0:
                return 1; // Flex
            case 1:
                return 2; // Rings
            case 2:
                return 3; // Globe
            case 3:
                return 4; // Cubes
            case 4:
                return 5; // Flex
        }
    } else if (ELLIADEN_TYPE(settings)) {
        switch (UICB_Value) {
            default:
            case 0:
                return 1; // Flex
            case 1:
                return 2; // Rings
            case 2:
                return 3; // Globe
            case 3:
                return 4; // Wavelength
        }
    } else {
        switch (UICB_Value) {
            default:
            case 0:
                return 1; // UltraSecure Key
            case 1:
                return 2; // Rings
            case 2:
                return 3; // Globe
            case 3:
                return 4; // Cubes
            case 4:
                return 5; // Flex
        }
    }
}

bool IgnoreImageData(struct settings_ *settings) {
    return settings->bEncodeOnly || settings->bLaminateOnly;
}

/*****************************************************************************
 *
 * GetChannelOption()
 *     Returns the colour format to be used for the given side of the card
 *     Checks the registry for Enduro family to determine if the dye film is
 *     different to the setting of the driver interface.  The appropriate dye
 *     film setting is returned.
 *
 *  Returns:
 *      Appropriate dye film setting
 *****************************************************************************/
static uint16_t GetChannelOption(struct settings_ *settings, int iFrontBack) {
    if (iFrontBack == FRONT || RIO_PRO_X_MODEL(settings)) {
        return settings->CardFront.ColourFormat;
    } else {
        return settings->CardBack.ColourFormat;
    }
}

/*****************************************************************************
 *  OutputPlaneDataSizeCommand()
 *      Send SZX commands to device.
 *
 *  Params:
 *      ulDataLen   The data length which is used as command parameter.
 *      ulColour    The colour to sent out. PLANE_Y, PLANE_M, PLANE_C, PLANE_K.
 *
 *  Returns:
 *      None
 *****************************************************************************/
void OutputPlaneDataSizeCommand(uint32_t ulDataLen, uint32_t ulColour) {
    switch (ulColour) {
        case PLANE_C:
            // Send command to set the length of the Cyan plane
            printf(CMD_STR_CYANSIZE, (long unsigned int)ulDataLen);
            break;

        case PLANE_M:
            // Send command to set the length of the Magenta plane
            printf(CMD_STR_MAGENTASIZE, (long unsigned int)ulDataLen);
            break;

        case PLANE_Y:
            // Send command to set the length of the Yellow plane
            printf(CMD_STR_YELLOWSIZE, (long unsigned int)ulDataLen);
            break;

        case PLANE_K:
            // Send command to set the length of the Black plane
            printf(CMD_STR_KSIZE, (long unsigned int)ulDataLen);
            break;

        default:
            // unknown colour
            break;
    }
}

/*****************************************************************************
 *  OutputHeaderCommand()
 *      Output header commands.
 *      The type of commands can be different depends on duplex setting.
 *      So this function requires which side (front or back) we are printing.
 *
 *  Parameters:
 *      pdev           Pointer to our PDEV
 *      header
 *      psettings
 *      bFrontPage     true = Front page (simplex or duplex);
 *                     false = Back page (simplex or duplex)
 *
 *  Returns:
 *      None
 *****************************************************************************/
void OutputHeaderCommand(PDEVDATA            pdev,
                         cups_page_header2_t header,
                         PSETTINGS           psettings,
                         bool                bFrontPage) {
    int  iCopyCount        = header.NumCopies;
    bool bOverCoat         = false;
    bool bUseWithLaminate  = false;
    bool bHoloKote         = false;
    int  iHoloKoteImage    = 0;
    int  iHoloKoteMap      = 0;
    bool bCustomKeyDisable = false;
    bool bHoloPatch        = false;
    bool bHoloPatchHole    = false;
    bool bArea_UserDefined = false;
    bool bHole_MagNormal   = false;
    bool bHole_MagWide     = false;
    bool bHole_ChipNormal  = false;
    bool bHole_ChipLarge   = false;
    bool bHole_UserDefined = false;
    int  iTileRotate       = 0;
    bool bMagUserDefined =
            (psettings->nTrackOptions == UICBVAL_Encoding_UserDefined);
    time_t         ltime;
    char           temp_string[20];
    int            i;
    uint32_t       dwPowerTemp;
    LPMAGTRACKINFO lpMagTrackInfo     = NULL;
    bool           bMagDataNotPresent = false;

    if ((psettings->bPerformVerification || psettings->bEncodeOnly) &&
        (!LAMINATOR_MODEL(psettings) || !psettings->bLaminateOnly) &&
        (pdev->magTrackInfo[MAG_TRACK_INDEX1].TrackData[0] == 0) &&
        (pdev->magTrackInfo[MAG_TRACK_INDEX2].TrackData[0] == 0) &&
        (pdev->magTrackInfo[MAG_TRACK_INDEX3].TrackData[0] == 0)) {
        bMagDataNotPresent = true;
    }

    // Send commands for leading characters and Start Of Header (SOH)
    for (i = 0; i < NUMBER_05S; i++) {
        printf(CMD_STR_LEADCHAR);
    }
    printf(CMD_STR_SOH);

    if (iCopyCount == 0)
        iCopyCount = 1;

    if (!ENDURO_OEM(psettings) && IgnoreImageData(psettings)) {
        // Send commands for printed and magnetic copies when "Magnetic encode
        // only" is selected
        printf(CMD_STR_NUMBEROFCOPY, 0);
        printf(CMD_STR_NUMBEROFCOPY_MAG, iCopyCount);
    } else {
        // Send command for hardware copies
        printf(CMD_STR_NUMBEROFCOPY, iCopyCount);
    }

    if ((bFrontPage) ||
        (ENDURO_OEM(psettings) && (psettings->Duplex == UICBVAL_Duplex_Back))) {
        // These commands are sent only for front page if image is double-sided

        if (psettings->Duplex == UICBVAL_Duplex_Back)
            printf(CMD_STR_BACKONLY);

        // Send commands for driver version
        printf(CMD_STR_DRIVERVERSION, DRV_VERSION);

        // If Enduro Family, add the firmware version to the header
        if (ENDURO_OEM(psettings)) {
            printf(CMD_STR_FWVERSION, psettings->es.sFirmwareVersion);
        }

        OutputComma();

        // Send commands for language
        // TODO
        printf(CMD_STR_LANGUAGE, "ENG" /*temp_string*/);

        // Put the current time and date in the header
        time(&ltime);
        sprintf(temp_string, "%X", (unsigned int)ltime);
        printf(CMD_STR_TIME_DATE, temp_string);

        // Insert the Card Size if it is CR79
        if (psettings->pageType == 2 && psettings->Printer != RIO_PRO_XTENDED &&
            psettings->Printer != PRIDENTOPROXTD &&
            psettings->Printer != PRICECARD_PRO_FLEX) {
            printf(CMD_STR_CR79);
        }

        if (ENDURO_OEM(psettings)) {
            if (!RIOPRO_TYPE(psettings)) {
                switch (psettings->nPrintSpeed) {
                    case UICBVAL_DefaultSpeed:
                        printf(CMD_STR_RESIN_QUALITY,
                               CMD_STR_RESIN_QUALITY_DEFAULT);
                        break;
                    case UICBVAL_HighSpeed:
                        printf(CMD_STR_RESIN_QUALITY,
                               CMD_STR_RESIN_QUALITY_FAST);
                        break;
                    case UICBVAL_HighQuality:
                        printf(CMD_STR_RESIN_QUALITY,
                               CMD_STR_RESIN_QUALITY_HIGH);
                        break;
                }
            }

            // Commands for Rewritable Cards......
            // Erase On
            if (!psettings->bEraseBeforePrint) {
                printf(CMD_STR_ERASE_ON);
                printf(CMD_STR_REWRITE_ERASE_AREA,
                       (unsigned int)psettings->EraseArea_Left,
                       (unsigned int)psettings->EraseArea_Bottom,
                       (unsigned int)psettings->EraseArea_Left +
                               (unsigned int)psettings->EraseArea_Width,
                       (unsigned int)psettings->EraseArea_Bottom +
                               (unsigned int)psettings->EraseArea_Height);
            }
            TRACE("ErasePower_Start: %u", psettings->ErasePower_Start);
            TRACE("ErasePower_End: %u", psettings->ErasePower_End);

            // Erase Start
            if (psettings->ErasePower_Start != DEFAULT_ERASE_START_POWER) {
                dwPowerTemp = (psettings->ErasePower_Start * 65535) / 100;
                printf(CMD_STR_ERASE_START, (unsigned int)dwPowerTemp);
            }

            // Erase End
            if (psettings->ErasePower_End != DEFAULT_ERASE_END_POWER) {
                dwPowerTemp = (psettings->ErasePower_End * 65535) / 100;
                printf(CMD_STR_ERASE_END, (unsigned int)dwPowerTemp);
            }

            // Erase Write
            if (psettings->WritePowerAdjustment != DEFAULT_ERASE_WRITE_POWER) {
                printf(CMD_STR_ERASE_WRITE, psettings->WritePowerAdjustment);
            }
        }

        switch (psettings->nDyeFilmOption) {
            case UICBVAL_DyeFilm_LC1:
            case UICBVAL_DyeFilm_AV1:
                printf(CMD_STR_LC1);
                break;
            case UICBVAL_DyeFilm_LC3:
                printf(CMD_STR_LC3);
                break;
            case UICBVAL_DyeFilm_LC6:
                printf(CMD_STR_LC6);
                break;
            case UICBVAL_DyeFilm_LC8:
                printf(CMD_STR_LC8);
                break;
            default:
            case UICBVAL_DyeFilm_Auto:
                // No command to be inserted - this is default and can only be
                // changed in Rio
                break;
        }

        // Magnetic Encoding Commands
        if (psettings->bJIS2Enabled) {
            printf(CMD_STR_JIS2_ENCODING);
        }

        if (!LAMINATOR_MODEL(psettings) || !psettings->bLaminateOnly) {
            uint32_t ulTrack = 0;

            if (RIO_OEM(psettings) || AVALON_MODEL(psettings) ||
                ENDURO_OEM(psettings)) {
                // Send magnetic encoding command and data to the device.
                if (psettings->bPerformVerification &&
                    bMagDataNotPresent == false) {
                    // Send command for Mag Verify On
                    printf(CMD_STR_MAGVERIFY);
                }
            }

            if ((RIO_OEM(psettings) && !bMagUserDefined) ||
                (RIOPRO_TYPE(psettings) && !bMagUserDefined) ||
                ENDURO_TYPE(psettings) || ENDURO_PLUS_TYPE(psettings) ||
                PRONTO_TYPE(psettings)) {
                switch (psettings->nCoercivity) {
                    case UICBVAL_Coercivity_HiCo:
                        // Send command for High Coercivity
                        printf(CMD_STR_HICOERCIVITY);
                        break;

                    case UICBVAL_Coercivity_LoCo:
                        // Send command for Low Coercivity
                        printf(CMD_STR_LOCOERCIVITY);
                        break;
                }
            }

            for (ulTrack = FIRST_MAG_TRACK; ulTrack <= LAST_MAG_TRACK;
                 ulTrack++) {
                lpMagTrackInfo = &pdev->magTrackInfo[ulTrack];

                if (lpMagTrackInfo->TrackData[0] != 0) {
                    printf(CMD_STR_STARTMAGDATA);

                    // Write the track number and comma from the track data
                    myWrite(lpMagTrackInfo->TrackData, 2);

                    if (!bMagUserDefined) {
                        WriteBitsPerChar(psettings, ulTrack);
                        WriteBitsPerInch(psettings, ulTrack);
                    }

                    myWrite(&lpMagTrackInfo->TrackData[2],
                            (uint32_t)(strlen(lpMagTrackInfo->TrackData) - 2));

                    OutputComma();
                }
            }
        }

        // only issue handfeed command if printer is not remote
        if (pdev->epPrinterRemote == false) {
            if (RIO_OEM(psettings) && psettings->HandFeed) {
                // Send command for hand feed paper source
                printf(CMD_STR_HANDFEED, CMD_STR_PARAM_ON);
            }
        }

        if (TANGO_MODEL(psettings) || LAMINATOR_MODEL(psettings)) {
            // Send command for setting reject on or off
            printf(CMD_STR_CARDREJECT,
                   psettings->CardReject ? CMD_STR_PARAM_ON :
                                           CMD_STR_PARAM_OFF);

            if (TANGO_MODEL(psettings)) {
                // Tango Specific options
                // Send command for selecting the card eject side
                printf(CMD_STR_EJECTSIDE,
                       psettings->EjectSide == UICBVAL_EjectSide_Front ?
                               FRONT_PAGE :
                               BACK_PAGE);
            } else {
                // Laminator Specific options
                // Insert Laminator Commands
                LaminatorCommand(pdev, psettings);
            }
        }
    }

    // Output the Printhead identifier
    if (RIO_OEM(psettings) || AVALON_MODEL(psettings)) {
        printf(CMD_STR_PRINTHEAD_RIO);
    } else if (ENDURO_OEM(psettings)) {
        switch (psettings->es.ePrintheadType) {
            default:
            case UICBVAL_PrintheadType_KGE2:
                printf(CMD_STR_PRINTHEAD_ENDURO_KGE2);
                break;
            case UICBVAL_PrintheadType_KEE1:
                printf(CMD_STR_PRINTHEAD_ENDURO_KEE1);
                break;
            case UICBVAL_PrintheadType_KEE4:
                printf(CMD_STR_PRINTHEAD_ENDURO_KEE4);
                break;
        }

    } else {
        printf(CMD_STR_PRINTHEAD_ALTO);
    }
    TRACE("FLUSHING OutputCommandHdr %u", bFrontPage);
    fflush(stdout);
    if (IgnoreImageData(psettings)) {
        // Send commands for end-of-header and end-of-page
        OutputFileSeparator();
        OutputETXCommand();

        // skip rest of job data since doing magnetic encoding or laminate only
        return;
    }

    // Add a comma after the Printhead Identifier as we are not doing mag encode
    // only
    OutputComma();

    if (RIO_OEM(psettings)) {
        printf(CMD_STR_RIOTANGO2);

        // Send command for ColourSure On/Off
        printf(CMD_STR_COLOURSURE,
               psettings->bColourSure ? CMD_STR_PARAM_ON : CMD_STR_PARAM_OFF);
    }

    if (RIOPRO_TYPE(psettings)) {
        // Send command for ColourSure On/Off
        printf(CMD_STR_COLOURSURE,
               psettings->nPrintSpeed ? CMD_STR_PARAM_ON : CMD_STR_PARAM_OFF);
    }

    if (psettings->Printer == RIO_PRO_XTENDED ||
        psettings->Printer == PRIDENTOPROXTD ||
        psettings->Printer == PRICECARD_PRO_FLEX) {
        // 1400 1654
        float fcx;
        int   icx;
        if (psettings->Printer == PRICECARD_PRO_FLEX) {
            if (psettings->pageType == 1) // extd mono card 140mm
                psettings->PaperHeight = 1400;
            else if (psettings->pageType == 3) // extd mono card 109mm
                psettings->PaperHeight = 1090;
            else if (psettings->pageType == 4) // extd mono card 128mm
                psettings->PaperHeight = 1280;

        } else {
            // just handle xtd for now
            if (psettings->pageType == 2) // extd colour card
                psettings->PaperHeight = 1080;
            else if (psettings->pageType == 3) // extd mono card 140mm
                psettings->PaperHeight = 1400;
            else if (psettings->pageType == 4) // extd mono card 109mm
                psettings->PaperHeight = 1090;
            else if (psettings->pageType == 5) // extd mono card 128mm
                psettings->PaperHeight = 1280;
        }
        fcx = (psettings->PaperHeight * (float)300) / (float)254;
        icx = (int)(fcx + 0.5f);

        // Send command to define the XXL Image Type & Size
        printf(CMD_STR_XXL_IMAGE_TYPE,
               2,
               (psettings->XXL_ImageType == UICBVAL_SingleImage) ? 1026 :
               (RIO_OEM(psettings)) ? pdev->epPaperXdots :
                                      (short)icx);
    }

    if (RIO_OEM(psettings) || AVALON_MODEL(psettings) ||
        ENDURO_OEM(psettings)) {
        // Send command for duplex on/off
        printf(CMD_STR_DUPLEX,
               pdev->bDuplex ? CMD_STR_PARAM_ON : CMD_STR_PARAM_OFF);
    }

    if (bFrontPage) {
        int iImageFormat = 0;

        // Back page image format
        if (pdev->bDuplex) {
            // Two sided
            iImageFormat = GetChannelOption(psettings, BACK);

            if (psettings->CardBack.PrintOvercoat &&
                !GetDataPresent(pdev, false)) {
                // No data on the back?
                printf(CMD_STR_BACKPAGEFORMAT, CMD_STR_PARAM_O_BAC);
            } else if ((psettings->CardBack.PrintOvercoat) ||
                       (RIO_PRO_X_MODEL(psettings) &&
                        psettings->CardFront.PrintOvercoat)) {
                // Send back page format command on the NEXT page (with
                // overcoat)
                switch (iImageFormat) {
                    case UICBVAL_YMCK_Back:
                    default:
                        // Send command for back side YMCK with overcoat (next
                        // page)
                        printf(CMD_STR_BACKPAGEFORMAT, CMD_STR_PARAM_CMYKO_BAC);
                        break;

                    case UICBVAL_YMC_Back:
                        // Send command for back side YMC with overcoat (next
                        // page)
                        printf(CMD_STR_BACKPAGEFORMAT, CMD_STR_PARAM_CMYO_BAC);
                        break;

                    case UICBVAL_KResin_Back:
                        // Send command for back side K-resin with overcoat
                        // (next page)
                        printf(CMD_STR_BACKPAGEFORMAT, CMD_STR_PARAM_KO_BAC);
                        break;
                }
            } else if (GetDataPresent(pdev, false)) {
                // Data on the back?
                // Send back page format command (with no overcoat)
                switch (iImageFormat) {
                    case UICBVAL_YMCK_Back:
                    default:
                        // Send command for back side YMCK with no overcoat
                        // (next page)
                        printf(CMD_STR_BACKPAGEFORMAT, CMD_STR_PARAM_CMYK_BAC);
                        break;

                    case UICBVAL_YMC_Back:
                        // Send command for back side YMC with no overcoat (next
                        // page)
                        printf(CMD_STR_BACKPAGEFORMAT, CMD_STR_PARAM_CMY_BAC);
                        break;

                    case UICBVAL_KResin_Back:
                        // Send command for back side K-resin with no overcoat
                        // (next page)
                        printf(CMD_STR_BACKPAGEFORMAT, CMD_STR_PARAM_K_BAC);
                        break;
                }
            }
        }

        // Send command to say that this is a front page
        printf(CMD_STR_CURRENTPAGE, FRONT_PAGE);

        // Data on the front?
        if (GetDataPresent(pdev, true)) {

            iImageFormat = GetChannelOption(psettings, FRONT);
            TRACE("data on the front :iImageFormat:%u", iImageFormat);
            // Send front page format command. Overcoat is determined below.
            switch (iImageFormat) {
                case UICBVAL_YMC_Front:
                    // Send command for front side YMC
                    printf(CMD_STR_IMAGEFORMAT, CMD_STR_PARAM_CMY);
                    break;

                case UICBVAL_YMCK_Front:
                default:
                    // Send command for front side YMCK
                    printf(CMD_STR_IMAGEFORMAT, CMD_STR_PARAM_CMYK);
                    break;

                case UICBVAL_KResin_Front:
                    // Send command for front side K-resin
                    printf(CMD_STR_IMAGEFORMAT, CMD_STR_PARAM_K);
                    break;
            }
        } else {
            // emulate a blank page to maintain wysiwyg
            printf(CMD_STR_IMAGEFORMAT, CMD_STR_PARAM_K);
        }
    } else {
        // bFrontPage == false
        int iImageFormat = GetChannelOption(psettings, BACK);

        // Send command to say that this is a back page
        printf(CMD_STR_CURRENTPAGE,
               ENDURO_OEM(psettings) &&
                               (psettings->Duplex == UICBVAL_Duplex_Back) ?
                       FRONT_PAGE :
                       BACK_PAGE);

        // Send back page format command. Overcoat is determined below.
        switch (iImageFormat) {
            case UICBVAL_YMC_Back:
                // Send command for back side YMC
                printf(CMD_STR_IMAGEFORMAT, CMD_STR_PARAM_CMY);
                break;

            case UICBVAL_YMCK_Back:
            default:
                // Send command for back side YMCK
                printf(CMD_STR_IMAGEFORMAT, CMD_STR_PARAM_CMYK);
                break;

            case UICBVAL_KResin_Back:
                // Send command for back side K-resin
                printf(CMD_STR_IMAGEFORMAT, CMD_STR_PARAM_K);
                break;
        }
    }

    // Send command to set page origin
    printf(CMD_STR_ORIGIN);

    // Send command to set page area
    printf(CMD_STR_IMAGESIZE, pdev->epPaperXdots, pdev->epPaperYdots);

    // Record the overcoat options so that they can be used to determine
    // commands that need to be sent (see later)
    if (bFrontPage) {
        bOverCoat = psettings->CardFront.PrintOvercoat;

        if (bOverCoat) {
            // Record Front Overcoat Options
            bHoloKote    = psettings->CardFront.HoloKote;
            iHoloKoteMap = psettings->CardFront.SecurityOptions.HoloKoteMap;

            if (RIO_OEM(psettings) || ENDURO_OEM(psettings)) {
                bCustomKeyDisable = psettings->CardFront.SecurityOptions
                                            .DisableCustomHoloKoteKey;
                bUseWithLaminate =
                        psettings->CardFront.SecurityOptions.UsewithLaminate;
                bHoloPatch = psettings->CardFront.HoloPatch;
                bHoloPatchHole =
                        psettings->CardFront.SecurityOptions.ColourHole;

                if (RIO_OEM(psettings)) {
                    // Rio/Tango front overcoat options
                    iHoloKoteImage =
                            psettings->CardFront.SecurityOptions.SecurityType +
                            1;
                } else if (RIOPRO_TYPE(psettings) || PRIDENTO_TYPE(psettings) ||
                           IDVILLE_TYPE(psettings)) {
                    // Rio Pro front overcoat options
                    iHoloKoteImage = RioProHKTValue(
                            psettings,
                            psettings->CardFront.SecurityOptions.SecurityType);
                } else {
                    // Other Enduro OEM front overcoat options
                    iHoloKoteImage =
                            psettings->CardFront.SecurityOptions.SecurityType +
                            1;
                    if (POLAROID_TYPE(psettings)) {
                        bCustomKeyDisable =
                                (psettings->CardFront.SecurityOptions
                                         .SecurityType !=
                                 UICBVAL_HoloKote_Polaroid_Enduro_Front);
                    }
                }
            } else {
                // AOTA Front overcoat options
                // Colour hole always on if holopatch is enabled
                bHoloPatch = bHoloPatchHole = psettings->CardFront.HoloPatch;

                if (AVALON_MODEL(psettings)) {
                    bUseWithLaminate = psettings->CardFront.SecurityOptions
                                               .UsewithLaminate;
                }
            }

            switch (psettings->CardFront.SecurityOptions.Rotation) {
                default:
                case UICBVAL_TileRotate_Front_0:
                    iTileRotate = 0;
                    break;
                case UICBVAL_TileRotate_Front_90:
                    iTileRotate = 90;
                    break;
                case UICBVAL_TileRotate_Front_180:
                    iTileRotate = 180;
                    break;
                case UICBVAL_TileRotate_Front_270:
                    iTileRotate = 270;
                    break;
            }

            bArea_UserDefined =
                    psettings->CardFront.OvercoatOptions.bUserDefined;

            switch (psettings->CardFront.OvercoatOptions.Holes) {
                case UICBVAL_MagStripeNormal:
                    bHole_MagNormal = true;
                    break;
                case UICBVAL_MagStripeWide:
                    bHole_MagWide = true;
                    break;
                case UICBVAL_ChipNormal:
                    bHole_ChipNormal = true;
                    break;
                case UICBVAL_ChipLarge:
                    bHole_ChipLarge = true;
                    break;
                case UICBVAL_UserDefined:
                    bHole_UserDefined = true;
                    break;
            }
        }
    } else {
        // bFrontPage == false (i.e. This is the back)
        bOverCoat = RIO_PRO_X_MODEL(psettings) ?
                            psettings->CardFront.PrintOvercoat :
                            psettings->CardBack.PrintOvercoat;

        if (bOverCoat) {
            // Record Back Overcoat Options
            bHoloKote = RIO_PRO_X_MODEL(psettings) ?
                                psettings->CardFront.HoloKote :
                                psettings->CardBack.HoloKote;
            iHoloKoteMap =
                    RIO_PRO_X_MODEL(psettings) ?
                            psettings->CardFront.SecurityOptions.HoloKoteMap :
                            psettings->CardBack.SecurityOptions.HoloKoteMap;

            if (RIO_OEM(psettings) || ENDURO_OEM(psettings)) {
                bCustomKeyDisable = RIO_PRO_X_MODEL(psettings) ?
                                            psettings->CardFront.SecurityOptions
                                                    .DisableCustomHoloKoteKey :
                                            psettings->CardBack.SecurityOptions
                                                    .DisableCustomHoloKoteKey;
                bUseWithLaminate  = RIO_PRO_X_MODEL(psettings) ?
                                            psettings->CardFront.SecurityOptions
                                                   .UsewithLaminate :
                                            psettings->CardBack.SecurityOptions
                                                   .UsewithLaminate;

                if (RIO_OEM(psettings)) {
                    // Rio/Tango back overcoat options
                    iHoloKoteImage =
                            psettings->CardBack.SecurityOptions.SecurityType +
                            1;
                } else if (RIOPRO_TYPE(psettings) || PRIDENTO_TYPE(psettings) ||
                           IDVILLE_TYPE(psettings)) {
                    // Rio Pro back overcoat options
                    iHoloKoteImage = RioProHKTValue(
                            psettings,
                            RIO_PRO_X_MODEL(psettings) ?
                                    psettings->CardFront.SecurityOptions
                                            .SecurityType :
                                    psettings->CardBack.SecurityOptions
                                            .SecurityType);

                } else {
                    // Other Enduro OEM back overcoat options
                    iHoloKoteImage =
                            psettings->CardBack.SecurityOptions.SecurityType +
                            1;
                    if (POLAROID_TYPE(psettings)) {
                        bCustomKeyDisable =
                                (psettings->CardBack.SecurityOptions
                                         .SecurityType !=
                                 UICBVAL_HoloKote_Polaroid_Enduro_Back);
                    }
                }
            } else {
                // AOTA Back overcoat options
                if (AVALON_MODEL(psettings)) {
                    bUseWithLaminate =
                            psettings->CardBack.SecurityOptions.UsewithLaminate;
                }
            }
            switch (RIO_PRO_X_MODEL(psettings) ?
                            psettings->CardFront.SecurityOptions.Rotation :
                            psettings->CardBack.SecurityOptions.Rotation) {
                default:
                case UICBVAL_TileRotate_Back_0:
                    iTileRotate = 0;
                    break;
                case UICBVAL_TileRotate_Back_90:
                    iTileRotate = 90;
                    break;
                case UICBVAL_TileRotate_Back_180:
                    iTileRotate = 180;
                    break;
                case UICBVAL_TileRotate_Back_270:
                    iTileRotate = 270;
                    break;
            }

            bArea_UserDefined =
                    RIO_PRO_X_MODEL(psettings) ?
                            psettings->CardFront.OvercoatOptions.bUserDefined :
                            psettings->CardBack.OvercoatOptions.bUserDefined;

            switch (RIO_PRO_X_MODEL(psettings) ?
                            psettings->CardFront.OvercoatOptions.Holes :
                            psettings->CardBack.OvercoatOptions.Holes) {
                case UICBVAL_MagStripeNormal:
                    bHole_MagNormal = true;
                    break;
                case UICBVAL_MagStripeWide:
                    bHole_MagWide = true;
                    break;
                case UICBVAL_ChipNormal:
                    bHole_ChipNormal = true;
                    break;
                case UICBVAL_ChipLarge:
                    bHole_ChipLarge = true;
                    break;
                case UICBVAL_UserDefined:
                    bHole_UserDefined = true;
                    break;
            }
        }
    }

    // Send Overcoat options to header
    if (!bOverCoat) {
        // Send command for not printing overcoat
        printf(CMD_STR_OVERCOAT, CMD_STR_PARAM_OFF);
    } else {
        // Send command for printing overcoat
        printf(CMD_STR_OVERCOAT, CMD_STR_PARAM_ON);

        // Send command for printing SecureShield/Use With Laminate
        printf(CMD_STR_USEWITHLAMINATE,
               bUseWithLaminate ? CMD_STR_PARAM_ON : CMD_STR_PARAM_OFF);

        // Send command for printing HoloKote
        printf(CMD_STR_HOLOKOTE,
               bHoloKote ? CMD_STR_PARAM_ON : CMD_STR_PARAM_OFF);

        if (bHoloKote || bUseWithLaminate || bHoloPatch) {
            if (RIO_OEM(psettings) || ENDURO_OEM(psettings)) {
                // Send Command for HoloKote Image Selection
                printf(CMD_STR_HOLOKOTE_IMAGE, iHoloKoteImage);

                // Send Command for Custom Key Disable
                printf(CMD_STR_CUSTOMKEY_DISABLE,
                       bCustomKeyDisable ? CMD_STR_PARAM_OFF :
                                           CMD_STR_PARAM_ON);

                if (RIO_OEM(psettings) || ENDURO_OEM(psettings)) {
                    // Send Command for HoloKoteMap
                    char sHoloKoteMap[7];
                    sprintf(sHoloKoteMap, "%06X", iHoloKoteMap);

                    printf(CMD_STR_HOLOKOTE_MAP, sHoloKoteMap);
                }

                // Send command for TileRotate (Rio/Tango/Enduro)
                switch (iTileRotate) {
                    case 0:
                    default:
                        printf(CMD_STR_TILEROTATE, CMD_STR_ROTATE_NONE);
                        break;

                    case 90:
                        printf(CMD_STR_TILEROTATE, CMD_STR_ROTATE_90);
                        break;

                    case 180:
                        printf(CMD_STR_TILEROTATE, CMD_STR_ROTATE_180);
                        break;

                    case 270:
                        printf(CMD_STR_TILEROTATE, CMD_STR_ROTATE_270);
                        break;
                }
            } else if (AVALON_MODEL(psettings)) {
                // Send command for TileRotate (Avalon)
                switch (iTileRotate) {
                    case 0:
                    default:
                        printf(CMD_STR_TILEROTATE, CMD_STR_PARAM_OFF);
                        break;

                    case 180:
                        printf(CMD_STR_TILEROTATE, CMD_STR_PARAM_ON);
                        break;
                }
            } else {
                // Send command for TileRotate (Alto/Opera/Tempo)
                switch (iTileRotate) {
                    case 0:
                    default:
                        printf(CMD_STR_TILEROTATE, CMD_STR_PARAM_OFF);
                        printf(CMD_STR_HOLOKOTE_ORIENT,
                               CMD_STR_HOLOKOTE_LANDSCAPE);
                        break;

                    case 90:
                        printf(CMD_STR_TILEROTATE, CMD_STR_PARAM_OFF);
                        printf(CMD_STR_HOLOKOTE_ORIENT,
                               CMD_STR_HOLOKOTE_PORTRAIT);
                        break;

                    case 180:
                        printf(CMD_STR_TILEROTATE, CMD_STR_PARAM_ON);
                        printf(CMD_STR_HOLOKOTE_ORIENT,
                               CMD_STR_HOLOKOTE_LANDSCAPE);
                        break;

                    case 270:
                        printf(CMD_STR_TILEROTATE, CMD_STR_PARAM_ON);
                        printf(CMD_STR_HOLOKOTE_ORIENT,
                               CMD_STR_HOLOKOTE_PORTRAIT);
                        break;
                }
            }
        }

        if (!bArea_UserDefined) {
            if (RIO_OEM(psettings) || AVALON_MODEL(psettings)) {
                printf(CMD_STR_OVERCOAT_RIO);
            } else if (ENDURO_OEM(psettings)) {
                printf(CMD_STR_OVERCOAT_ENDURO);
            } else {
                printf(CMD_STR_OVERCOAT_ALTO);
            }
        } else {
            if (bFrontPage) {
                if (AreaHoleActive(psettings, FRONT_AREA1)) {
                    printf(CMD_STR_OVERCOAT_USER,
                           (long unsigned int)psettings->CardFront.Area[0].left,
                           (long unsigned int)psettings->CardFront.Area[0]
                                   .bottom,
                           (long unsigned int)psettings->CardFront.Area[0]
                                           .left +
                                   psettings->CardFront.Area[0].width,
                           (long unsigned int)psettings->CardFront.Area[0]
                                           .bottom +
                                   psettings->CardFront.Area[0].height);
                }
                if (AreaHoleActive(psettings, FRONT_AREA2)) {
                    printf(CMD_STR_OVERCOAT_USER,
                           (long unsigned int)psettings->CardFront.Area[1].left,
                           (long unsigned int)psettings->CardFront.Area[1]
                                   .bottom,
                           (long unsigned int)psettings->CardFront.Area[1]
                                           .left +
                                   psettings->CardFront.Area[1].width,
                           (long unsigned int)psettings->CardFront.Area[1]
                                           .bottom +
                                   psettings->CardFront.Area[1].height);
                }
            } else {
                if (AreaHoleActive(psettings, BACK_AREA1)) {
                    printf(CMD_STR_OVERCOAT_USER,
                           (long unsigned int)psettings->CardBack.Area[0].left,
                           (long unsigned int)psettings->CardBack.Area[0]
                                   .bottom,
                           (long unsigned int)psettings->CardBack.Area[0].left +
                                   psettings->CardBack.Area[0].width,
                           (long unsigned int)psettings->CardBack.Area[0]
                                           .bottom +
                                   psettings->CardBack.Area[0].height);
                }
                if (AreaHoleActive(psettings, BACK_AREA2)) {
                    printf(CMD_STR_OVERCOAT_USER,
                           (long unsigned int)psettings->CardBack.Area[1].left,
                           (long unsigned int)psettings->CardBack.Area[1]
                                   .bottom,
                           (long unsigned int)psettings->CardBack.Area[1].left +
                                   psettings->CardBack.Area[1].width,
                           (long unsigned int)psettings->CardBack.Area[1]
                                           .bottom +
                                   psettings->CardBack.Area[1].height);
                }
            }
        }

        if (bHoloPatch) {
            // Send command for HoloPatch
            for (i = 0; i < 24; i++)
                if (psettings->CardFront.SecurityOptions.HoloKotePatch &
                    (1 << i))
                    break;

            printf(CMD_STR_HOLOPATCHPOS, i + 1);

            // Send command for HoloPatchHole
            printf(CMD_STR_HOLOPATCHHOLE,
                   bHoloPatchHole ? CMD_STR_PARAM_ON : CMD_STR_PARAM_OFF);
        }

        if (bHole_MagNormal) {
            printf(CMD_STR_MAG_STRIP_NORMAL);
        } else if (bHole_MagWide) {
            printf(CMD_STR_MAG_STRIP_WIDE);
        } else if (bHole_ChipNormal) {
            printf(CMD_STR_CHIP_NORMAL);
        } else if (bHole_ChipLarge) {
            printf(CMD_STR_CHIP_LARGE);
        } else if (bHole_UserDefined) {
            if (bFrontPage) {
                if (AreaHoleActive(psettings, FRONT_HOLE1)) {
                    printf(CMD_STR_HOLE_USER_DEFINED,
                           (long unsigned int)psettings->CardFront.Hole[0].left,
                           (long unsigned int)psettings->CardFront.Hole[0]
                                   .bottom,
                           (long unsigned int)psettings->CardFront.Hole[0]
                                           .left +
                                   psettings->CardFront.Hole[0].width,
                           (long unsigned int)psettings->CardFront.Hole[0]
                                           .bottom +
                                   psettings->CardFront.Hole[1].height);
                }
                if (AreaHoleActive(psettings, FRONT_HOLE2)) {
                    printf(CMD_STR_HOLE_USER_DEFINED,
                           (long unsigned int)psettings->CardFront.Hole[1].left,
                           (long unsigned int)psettings->CardFront.Hole[1]
                                   .bottom,
                           (long unsigned int)psettings->CardFront.Hole[1]
                                           .left +
                                   psettings->CardFront.Hole[1].width,
                           (long unsigned int)psettings->CardFront.Hole[1]
                                           .bottom +
                                   psettings->CardFront.Hole[1].height);
                }
            } else {
                if (AreaHoleActive(psettings, BACK_HOLE1)) {
                    printf(CMD_STR_HOLE_USER_DEFINED,
                           (long unsigned int)psettings->CardBack.Hole[0].left,
                           (long unsigned int)psettings->CardBack.Hole[0]
                                   .bottom,
                           (long unsigned int)psettings->CardBack.Hole[0].left +
                                   psettings->CardBack.Hole[0].width,
                           (long unsigned int)psettings->CardBack.Hole[0]
                                           .bottom +
                                   psettings->CardBack.Hole[0].height);
                }
                if (AreaHoleActive(psettings, BACK_HOLE2)) {
                    printf(CMD_STR_HOLE_USER_DEFINED,
                           (long unsigned int)psettings->CardBack.Hole[1].left,
                           (long unsigned int)psettings->CardBack.Hole[1]
                                   .bottom,
                           (long unsigned int)psettings->CardBack.Hole[1].left +
                                   psettings->CardBack.Hole[1].width,
                           (long unsigned int)psettings->CardBack.Hole[1]
                                           .bottom +
                                   psettings->CardBack.Hole[1].height);
                }
            }
        }
    }

    // Send Power commands to header...
    // ...Send command for YMC printhead power
    // send command for ICC profile selected.. informs firmware of base heat to
    // apply
    // always send to all models for tech support diagnosis

    printf(CMD_STR_ICC, psettings->nColourCorrection);

    if (psettings->nColourCorrection == UICBVAL_ColourCorrection_ICC_Internal) {
        printf(CMD_STR_PH_YMC, psettings->nPrintHeadPower_YMC);
    } else if (psettings->nPrintHeadPower_YMC != DEFAULT_PRINTHEAD_POWER_YMC) {
        printf(CMD_STR_PH_YMC, psettings->nPrintHeadPower_YMC);
    }

    if (psettings->nPrintHeadPower_BlackResin !=
        DEFAULT_PRINTHEAD_POWER_RESIN) {
        //...Send command for K-resin printhead power
        printf(CMD_STR_PH_K, psettings->nPrintHeadPower_BlackResin);
    }

    if (psettings->nPrintHeadPower_Overcoat !=
        DEFAULT_PRINTHEAD_POWER_OVERCOAT) {
        //...Send command for overcoat printhead power
        printf(CMD_STR_PH_OVER, psettings->nPrintHeadPower_Overcoat);
    }

    if (psettings->nPrintHeadPosition != 0) {
        //...Send command for printhead position
        printf(CMD_STR_PH_POSITION, psettings->nPrintHeadPosition);
    }

    if (psettings->nImagePosition_Start != 0) {
        //...Send command for image start position
        if (psettings->OEM != OEM_ENDURO) {
            printf(CMD_STR_IS_POSITION,
                   DEFAULT_IMAGE_START_POSITION -
                           psettings->nImagePosition_Start);
        } else {
            printf(CMD_STR_IS_POSITION,
                   DEFAULT_IMAGE_START_POSITION +
                           psettings->nImagePosition_Start);
        }
    }

    if (psettings->nImagePosition_End != 0) {
        //...Send command for image end position
        if (psettings->OEM != OEM_ENDURO)
            printf(CMD_STR_IE_POSITION,
                   DEFAULT_IMAGE_END_POSITION - psettings->nImagePosition_End);
        else
            printf(CMD_STR_IE_POSITION,
                   DEFAULT_IMAGE_END_POSITION + psettings->nImagePosition_End);
    }

    // Handle ~0 commands that have come in from an application.  Just add
    // them into the header
    if (ENDURO_OEM(psettings)) {
        lpMagTrackInfo = bFrontPage ? &pdev->magTrackInfo[MAG_TRACK_INDEX0] :
                                      &pdev->magTrackInfo[MAG_TRACK_INDEX4];

        if (lpMagTrackInfo->TrackData[0] != 0) {
            // Do not include ? on the end if it exists
            i = strlen(lpMagTrackInfo->TrackData) - 2;
            if (lpMagTrackInfo->TrackData[strlen(lpMagTrackInfo->TrackData) -
                                          1] == '?') {
                i--;
            }
            if (i > 255)
                i = 255;
            myWrite(&lpMagTrackInfo->TrackData[2], i);
            OutputComma();
        }
    }
    return;
}

/*****************************************************************************
 *  OutputPlaneCommand()
 *      Send plane data and command.
 *
 *  Parameters:
 *      pdev         Pointer to our PDEV
 *      settings
 *      pPlane       Pointer to output plane data
 *      ulColour     The colour to send out. PLANE_Y, PLANE_M, PLANE_C, PLANE_K
 *      ulOutBufLen  The data length in pPlane
 *
 *  Returns:
 *      None
 *****************************************************************************/
void OutputPlaneCommand(PDEVDATA          pdev,
                        struct settings_ *settings,
                        uint8_t *         pPlane,
                        uint32_t          ulColour,
                        uint32_t          ulOutBufLen) {

    if (settings->OEM == OEM_PRO360) {
        TRACE("Calling myWritecs(), ulOutBufLen = %d", ulOutBufLen);
        myWritecs(pdev, (char *)pPlane, ulOutBufLen, 0);
    } else {
        // Send the plane data
        myWrite((char *)pPlane, ulOutBufLen);
    }

    if (settings->OEM != OEM_HELIX && settings->OEM != OEM_PRO360) {
        // Send command to complete the plane data
        OutputFileSeparator();

        // Colour
        switch (ulColour) {
            case PLANE_C:
                printf(CMD_STR_OUTPUT_C);
                break;
            case PLANE_M:
                printf(CMD_STR_OUTPUT_M);
                break;
            case PLANE_Y:
                printf(CMD_STR_OUTPUT_Y);
                break;
            case PLANE_K:
                printf(CMD_STR_OUTPUT_K);
                break;
        }
    }
}

/*****************************************************************************
 *  OutputHelixHeaderCommand
 *
 *  Parameters:
 *      pdev         Pointer to our PDEV
 *      header
 *      pSettings
 *      bFrontPage   true = Front page (simplex or duplex);
 *                   false = Back page (simplex or duplex)
 *
 *  Returns:
 *      None
 *****************************************************************************/
void OutputHelixHeaderCommand(PDEVDATA            pdev,
                              cups_page_header2_t header,
                              PSETTINGS           psettings,
                              bool                bFrontPage) {
    HDR_JOB        jobHeader;
    CARD_OPTIONS   cardOptions;
    LPSPOOLMEMINFO lpSplMemFront = NULL;
    LPSPOOLMEMINFO lpSplMemBack  = NULL;

    int  iCopyCount = header.NumCopies;
    int  ulColour;
    bool bBlankFrontPage   = true;
    bool bBlankBackPage    = true;
    bool bSendMagEncodeJob = false;
    memset((uint8_t *)&jobHeader, 0, sizeof(HDR_JOB));

    // see if any planes actually have data so we know whether to encode only..
    for (ulColour = 0; ulColour < 4; ulColour++) {
        lpSplMemFront = &pdev->lpSplMemFront[ulColour];
        lpSplMemBack  = &pdev->lpSplMemBack[ulColour];

        if (lpSplMemFront->lpBuffer != NULL && lpSplMemFront->bDataInPlane) {
            bBlankFrontPage = false;
        }

        if (lpSplMemBack->lpBuffer != NULL && lpSplMemBack->bDataInPlane) {
            bBlankBackPage = false;
        }
    }

    memcpy((uint8_t *)&jobHeader.magic, (uint8_t *)"0PT1MA00", 8);

    jobHeader.job_id = (JOB_ID)3;
    jobHeader.size   = sizeof(HDR_JOB) + sizeof(CARD_OPTIONS) +
                     ((pdev->bDuplex) ? PREVIEW_SIZE * 2 : PREVIEW_SIZE);
    jobHeader.job_sub_id = (pdev->bDuplex) ?
                                   POF_SIDE_1_PREVIEW | POF_SIDE_2_PREVIEW :
                                   POF_SIDE_1_PREVIEW;

    if (pdev->bDuplex) {
        if (psettings->CardFront.HoloKote) {
            jobHeader.job_sub_id |= POF_SIDE_1O;
        }
        if (psettings->CardBack.HoloKote) {
            jobHeader.job_sub_id |= POF_SIDE_2O;
        }
    } else {
        if (bFrontPage && psettings->CardFront.HoloKote) {
            jobHeader.job_sub_id |= POF_SIDE_1O;
        }
        if (bFrontPage == false && psettings->CardBack.HoloKote) {
            jobHeader.job_sub_id |= POF_SIDE_1O;
        }
    }

    if (psettings->ISO7810 && bSendMagEncodeJob == false &&
        pdev->bDuplex == false) {
        jobHeader.job_sub_id |= POF_FLATTEN_CARD;
    }

    // which planes actually have data
    for (ulColour = 0; ulColour < 4; ulColour++) {
        lpSplMemFront = &pdev->lpSplMemFront[ulColour];
        lpSplMemBack  = &pdev->lpSplMemBack[ulColour];

        if (lpSplMemFront->lpBuffer != NULL && lpSplMemFront->bDataInPlane) {
            if (ulColour == 0) {
                // yellow
                jobHeader.job_sub_id |= POF_SIDE_1Y;
            } else if (ulColour == 1) {
                // magenta
                jobHeader.job_sub_id |= POF_SIDE_1M;
            } else if (ulColour == 2) {
                // cyan
                jobHeader.job_sub_id |= POF_SIDE_1C;
            } else {
                // resin
                jobHeader.job_sub_id |= POF_SIDE_1K;
            }

            jobHeader.size += BUF_SIZE_300DPI;
            bBlankFrontPage = false;
        }

        if (lpSplMemBack->lpBuffer != NULL && lpSplMemBack->bDataInPlane) {
            if (pdev->bDuplex) {
                if (ulColour == 0) {
                    // yellow
                    jobHeader.job_sub_id |= POF_SIDE_2Y;
                } else if (ulColour == 1) {
                    // magenta
                    jobHeader.job_sub_id |= POF_SIDE_2M;
                } else if (ulColour == 2) {
                    // cyan
                    jobHeader.job_sub_id |= POF_SIDE_2C;
                } else {
                    // resin
                    jobHeader.job_sub_id |= POF_SIDE_2K;
                }
            } else {
                // set the job header to flag side 1 for back only
                if (ulColour == 0) {
                    // yellow
                    jobHeader.job_sub_id |= POF_SIDE_1Y;
                    bBlankFrontPage = false;
                } else if (ulColour == 1) {
                    // magenta
                    jobHeader.job_sub_id |= POF_SIDE_1M;
                    bBlankFrontPage = false;
                } else if (ulColour == 2) {
                    // cyan
                    jobHeader.job_sub_id |= POF_SIDE_1C;
                    bBlankFrontPage = false;
                } else {
                    // resin
                    jobHeader.job_sub_id |= POF_SIDE_1K;
                    bBlankFrontPage = false;
                }
            }
            jobHeader.size += BUF_SIZE_300DPI;
            bBlankBackPage = false;
        }
    }

    // ensure we send something for a blank page
    if (bBlankFrontPage) {
        jobHeader.job_sub_id |= POF_SIDE_1K;
        jobHeader.size += BUF_SIZE_300DPI;
    }
    if (pdev->bDuplex && bBlankBackPage) {
        jobHeader.job_sub_id |= POF_SIDE_2K;
        jobHeader.size += BUF_SIZE_300DPI;
    }

    jobHeader.version = COMMS_VERSION;
    time((time_t *)&jobHeader.timestamp);
    myWrite((char *)&jobHeader, sizeof(HDR_JOB));

    // now send a CardOptions struct
    memset((uint8_t *)&cardOptions, 0, sizeof(CARD_OPTIONS));
    cardOptions.num_copies = iCopyCount;
    mc_strncpy((char *)&cardOptions.driver_version,
               DRV_VERSION,
               sizeof(cardOptions.driver_version));
    mc_strncpy((char *)&cardOptions.driver_language,
               "ENG",
               sizeof(cardOptions.driver_language));

    cardOptions.rotate_before_print =
            (pdev->bDuplex == false && bFrontPage == false) ? 1 : 0;
    cardOptions.rotate_before_lam                         = 0;
    cardOptions.disable_eject                             = 0;
    cardOptions.side1_options.laminate_target_temperature = 64;
    cardOptions.side2_options.laminate_target_temperature = 64;

    if (pdev->bDuplex) {
        if (psettings->CardFront.HoloKote) {
            cardOptions.side1_options.holokote_index =
                    psettings->CFHolokoteSlot;
            cardOptions.side1_options.holokote_orientation =
                    (psettings->CardFront.SecurityOptions.Rotation) ? 180 : 0;
        }
        if (psettings->CardBack.HoloKote) {
            cardOptions.side2_options.holokote_index =
                    psettings->CBHolokoteSlot;
            cardOptions.side2_options.holokote_orientation =
                    (psettings->CardBack.SecurityOptions.Rotation) ? 180 : 0;
        }
    } else {
        if (bFrontPage && psettings->CardFront.HoloKote) {
            cardOptions.side1_options.holokote_index =
                    psettings->CFHolokoteSlot;
            cardOptions.side1_options.holokote_orientation =
                    (psettings->CardFront.SecurityOptions.Rotation) ? 180 : 0;
        }
        if (bFrontPage == false && psettings->CardBack.HoloKote) {
            cardOptions.side1_options.holokote_index =
                    psettings->CBHolokoteSlot;
            cardOptions.side1_options.holokote_orientation =
                    (psettings->CardBack.SecurityOptions.Rotation) ? 180 : 0;
        }
    }
    TRACE("POWERYMC:%d", -5000 + (psettings->nHLXPrintHeadPowerYMC * 100));
    cardOptions.side1_options.power_adjust[PANEL_YELLOW] =
            cardOptions.side1_options.power_adjust[PANEL_MAGENTA] =
                    cardOptions.side1_options.power_adjust[PANEL_CYAN] =
                            -5000 + (psettings->nHLXPrintHeadPowerYMC * 100);
    cardOptions.side1_options.power_adjust[PANEL_RESIN] =
            -5000 + (psettings->nHLXPrintHeadPowerK * 100);
    cardOptions.side1_options.power_adjust[PANEL_OVERCOAT] =
            -5000 + (psettings->nHLXPrintHeadPowerOvercoat * 100);

    cardOptions.side2_options.power_adjust[PANEL_YELLOW] =
            cardOptions.side2_options.power_adjust[PANEL_MAGENTA] =
                    cardOptions.side2_options.power_adjust[PANEL_CYAN] =
                            -5000 + (psettings->nHLXPrintHeadPowerYMC * 100);
    cardOptions.side2_options.power_adjust[PANEL_RESIN] =
            -5000 + (psettings->nHLXPrintHeadPowerK * 100);
    cardOptions.side2_options.power_adjust[PANEL_OVERCOAT] =
            -5000 + (psettings->nHLXPrintHeadPowerOvercoat * 100);

    cardOptions.side1_options.image_position_adjust_x = 0;
    cardOptions.side1_options.image_position_adjust_x = 0;
    cardOptions.side2_options.image_position_adjust_y = 0;
    cardOptions.side2_options.image_position_adjust_y = 0;
    myWrite((char *)&cardOptions, sizeof(CARD_OPTIONS));
}




/*******************************************************************************
 *      GetDuplexInfo()
 *      This function finds if the page is going to be duplexed and which side.
 *
 *  Parameters:
 *      pdev           Pointer to our PDEV
 *      settings
 *      iPageNumber    Current page number
 *
 *  Returns:
 *      None
 *******************************************************************************/
void GetDuplexInfo(PDEVDATA pdev, struct settings_ *settings, int iPageNumber) {

    // TODO:where we getting this from
    // then????//GetUICBScratchProperties(pdev->lpdm)->iTotalJobPages;
    int iLastLogicalPage = settings->iTotalJobPages;
    int iDuplex          = settings->Duplex;

    // note: if direct printing is enabled this will fail and will rely on
    // duplex set in application
    if (settings->Printer == RIO_PRO_XTENDED ||
        settings->Printer == PRIDENTOPROXTD ||
        settings->Printer == PRICECARD_PRO_FLEX) {
        TRACE("settings->XXL_ImageType: %u", settings->XXL_ImageType);
        if (settings->XXL_ImageType == UICBVAL_DoubleImage) {
            iDuplex = UICBVAL_Duplex_BothSides;
        }
    }

    if (iDuplex == UICBVAL_Duplex_BothSides) {
       TRACE("iDuplex: %u, iPageNumber: %u", iDuplex, iPageNumber);

        // Print direct to printer enabled so we have no detail of the number of
        // pages in the document
        if (iLastLogicalPage == 0) {
            pdev->bFrontPage = ((iPageNumber + 2) % 2) > 0;
            pdev->bDuplex    = true;
            TRACE("pdev->bFrontPage: %u, pdev->bDuplex: %u",
                  pdev->bFrontPage,
                  pdev->bDuplex);
            return;
        }

        if (iLastLogicalPage > 1) {
            pdev->bFrontPage = ((iPageNumber + 2) % 2) > 0;

            // We have more than 2 pages left so it is going to be duplexed
            if (iLastLogicalPage == iPageNumber && pdev->bFrontPage) {
                // But if only a page left, we treat it as simplex
                pdev->bDuplex = false;
            } else {
                pdev->bDuplex = true;
            }
        } else {
            // iLastLogicalPage <= 1
            pdev->bFrontPage = true;
            pdev->bDuplex    = false;
        }
    }

    else {
        // iDuplex != UICBVAL_Duplex_BothSides
        pdev->bDuplex = false;

        // Even simplex case, we still to know if it is back or front on UI
        // Because some option such as ribbon type can be different
        if (iDuplex == UICBVAL_Duplex_Back) {
            pdev->bFrontPage = false;
        } else {
            pdev->bFrontPage = true;
        }
    }
}

/*******************************************************************************
 *  DetectEdge()
 *      Search through the K-resin surface and decide the pixels
 *      to be modified at DetectAdjacentColour
 *
 *  Returns:
 *      None
 *******************************************************************************/
void DetectEdge(PDEVDATA pdev) {
    uint8_t *lpBlack = NULL;
    uint8_t *lpTemp  = NULL;
    uint8_t *lpHalo  = NULL;
    int32_t  x       = 0;
    int32_t  y       = 0;
    int32_t  TblX    = 0;
    int32_t  TblY    = 0;
    int32_t  StartX  = 0;
    int32_t  StartY  = 0;
    int32_t  EndX    = 0;
    int32_t  EndY    = 0;
    bool     bHalo   = false;
    POINTL   ptlPosition[MAX_HALO_WIDTH][MAX_HALO_WIDTH] = {0};
#define HALO_RADIUS pdev->iHaloRadius

    /******************************************************************************
     * Create matrix to get the pointer of the surrounding pixels.
     * If HALO_WIDTH == 3, the table will be...
     *
     *    (-1, -1) | (0, -1)  | (1, -1)
     *  ----------------------------------
     *    (-1,  0) | (0,  0)  | (1,  0)
     *  ----------------------------------
     *    (-1,  1) | (0,  1)  | (1,  1)
     *
     *******************************************************************************/
    for (TblY = 0; TblY < HALO_WIDTH; TblY++) {
        for (TblX = 0; TblX < HALO_WIDTH; TblX++) {
            if (TblX < HALO_RADIUS) {
                ptlPosition[TblX][TblY].x = -(HALO_RADIUS - TblX);
            } else if (TblX == HALO_RADIUS) {
                ptlPosition[TblX][TblY].x = 0;
            } else if (TblX > HALO_RADIUS) {
                ptlPosition[TblX][TblY].x = TblX - HALO_RADIUS;
            }

            if (TblY < HALO_RADIUS) {
                ptlPosition[TblX][TblY].y = -(HALO_RADIUS - TblY);
            } else if (TblY == HALO_RADIUS) {
                ptlPosition[TblX][TblY].y = 0;
            } else if (TblY > HALO_RADIUS) {
                ptlPosition[TblX][TblY].y = TblY - HALO_RADIUS;
            }
        }
    }

    for (y = 0; y < pdev->yImage; y++) {
        // Set up pointers to pixels in the Black and Halo planes
        lpBlack = pdev->lpPageBuffer[PLANE_K] + (pdev->ulCMYInPlaneWidth * y);
        lpHalo = pdev->lpPageBuffer[PLANE_HALO] + (pdev->ulCMYInPlaneWidth * y);

        // NOTE: lpHalo can be 1bpp to save memory and performance.

        // Decide the range to refer using a table.
        if (y < HALO_RADIUS) {
            // top
            StartY = HALO_RADIUS - y;
            EndY   = HALO_WIDTH - 1;
        } else if (pdev->yImage - (y + 1) < HALO_RADIUS) {
            // bottom
            StartY = 0;
            EndY   = pdev->yImage - (y + 1) + HALO_RADIUS;
        } else {
            // middle
            StartY = 0;
            EndY   = HALO_WIDTH - 1;
        }

        for (x = 0; x < (int32_t)pdev->ulCMYInPlaneWidth; x++) {
            /*******************************************************************************
             * Test the darkness of this pixel
             *
             * A threshold could be used in the following if (instead of testing
             *for white) so that
             * light grey pixels are not included in the halo plane either
             *******************************************************************************/

            if (*lpBlack == 0) {
                // Since this pixel is white it should not form part of a halo
                // for edge detection
                *lpHalo = 0x0;
            } else {
                // This pixel is black or greyscale

                if (x < HALO_RADIUS) {
                    // left edge
                    StartX = HALO_RADIUS - x;
                    EndX   = HALO_WIDTH - 1;
                } else if ((int32_t)pdev->ulCMYInPlaneWidth - (x + 1) <
                           HALO_RADIUS) {
                    // right edge
                    StartX = 0;
                    EndX   = pdev->ulCMYInPlaneWidth - (x + 1) + HALO_RADIUS;
                } else {
                    // middle
                    StartX = 0;
                    EndX   = HALO_WIDTH - 1;
                }

                // Start with the assumption that we are not near to a black
                // boundary
                bHalo = false;

                // Look around the pixel ... if it is different object or white.
                for (TblY = StartY; TblY <= EndY; TblY++) {
                    for (TblX = StartX; TblX <= EndX; TblX++) {
                        // Test a surrounding pixel in the black plane
                        lpTemp = lpBlack +
                                 ((int32_t)pdev->ulCMYInPlaneWidth *
                                            ptlPosition[TblX][TblY].y) +
                                 ptlPosition[TblX][TblY].x;

                        if (*lpTemp == 0x0) {
                            // We have found a surrounding pixel that is white,
                            // so flag this as a halo point
                            // and break out of the x loop
                            bHalo = true;
                            break;
                        }
                    }

                    if (bHalo) {
                        // We already know that this is a halo point, so break
                        // out of the y loop
                        break;
                    }
                }

                if (bHalo) {
                    *lpHalo = 0xff; // This is a halo pixel
                } else {
                    *lpHalo = 0x0; // This is NOT a halo pixel
                }
            }

            // Move on to the next pixel in the x-scanline for both planes
            lpBlack++;
            lpHalo++;
        }
    }

    return;
}

/*******************************************************************************
 *  ReplaceYMCPixelColour()
 *      Calculate the average value of surrounding 8 coloured pixels.
 *
 *  Parameters:
 *      lpPrev       The pixel directly above the one being processed
 *      lpCurrent    The pixel to be processed
 *      lpNext       The pixel directly below the one being processed
 *      bRight       true = This is the right pixel on the strip
 *
 *  Returns:
 *      None
 *******************************************************************************/
void ReplaceYMCPixelColour(uint8_t *lpPrev,
                           uint8_t *lpCurrent,
                           uint8_t *lpNext) {
    uint32_t ulPixel = 0;
    uint32_t ulDiv   = 0;

    /*******************************************************************************
     * For this pixel, calculate a weighted average of the valid surrounding
     *colours
     * Surrounding pixels must be non-white, and within scanline bounds to be
     *included
     * in this average
     *******************************************************************************/
    if (lpPrev != NULL) {
        if (*lpPrev != 0) {
            ulPixel += *lpPrev;
            ulDiv++;
        }
        lpPrev++;
    }
    if (lpNext != NULL) {
        if (*lpNext != 0) {
            ulPixel += *lpNext;
            ulDiv++;
        }
        lpNext++;
    }

    if (ulDiv != 0) {
        *lpCurrent = (uint8_t)(ulPixel / ulDiv);
    }
}

/*******************************************************************************
 *      CreateKPrintingObject()
 *      Create strip obj and halftone obj.
 *
 *  Returns:
 *      false if it fails to create objects.
 *******************************************************************************/
bool CreateKPrintingObject(PDEVDATA            pdev,
                           struct settings_ *  settings,
                           cups_page_header2_t header) {
    long     lWidth  = 0;
    long     lHeight = 0;
    uint32_t cb      = 0;

    if ((pdev->eChannelOption == UICBVAL_YMCK) ||
        (pdev->eChannelOption == UICBVAL_KResin)) {
        if (HELIX_OEM(settings)) {
            lWidth  = pdev->epPaperXdots;
            lHeight = pdev->epPaperYdots;
        } else {
            // create a kplane
            if (header.cupsHeight == 1016 || header.cupsHeight == 991 ||
                header.cupsHeight == 1654) {
                lWidth  = pdev->epPaperYdots;
                lHeight = pdev->epPaperXdots;
            } else {
                lWidth  = pdev->epPaperXdots;
                lHeight = pdev->epPaperYdots;
            }
        }
        // Create masking surface 8bpp duping the 24bpp orientation
        cb                = (lWidth * 3) * lHeight;
        pdev->lpKSrcStrip = malloc(cb);
        if (pdev->lpKSrcStrip == 0) {
            return false;
        }
        memset(pdev->lpKSrcStrip, 0x00, cb);

        // if we are landscape create a portrait masking surface
        pdev->lpKPSrcStrip = malloc(cb);
        if (pdev->lpKPSrcStrip == 0) {
            return false;
        }
        memset(pdev->lpKPSrcStrip, 0x00, cb);

        // Create destination surface 8bpp
        pdev->lpKDstStrip = malloc(cb);
        if (pdev->lpKDstStrip == 0) {
            return false;
        }
        memset(pdev->lpKDstStrip, 0x00, cb);
    }

    // Create Halftone 1bpp monochrome surface
    cb               = lWidth * lHeight;
    if (cb > 0) {
    pdev->lpHalftone = malloc(cb);
    }

    if (pdev->lpHalftone == NULL) {
        return false;
    }
    memset(pdev->lpHalftone, 0x00, cb);

    return true;
}

/*******************************************************************************
 *  DeleteOutputBuffer()
 *      Clean up buffers which were allocated at InitializeOutputBuffer.
 *      Need to be called in the end of a page.
 *
 *  Returns:
 *      None
 *******************************************************************************/
void DeleteOutputBuffer(PDEVDATA pdev) {

    if (pdev->lpCMYIn[0] != 0) {
        free(pdev->lpCMYIn[0]);
        pdev->lpCMYIn[0] = 0;
    }

    if (pdev->lpCMYOut[0] != 0) {
        free(pdev->lpCMYOut[0]);
        pdev->lpCMYOut[0] = 0;
    }

    if (pdev->lpKOut != 0) {
        free(pdev->lpKOut);
        pdev->lpKOut = 0;
    }

    if (pdev->lpPageBuffer[0] != 0) {
        free(pdev->lpPageBuffer[0]);
        pdev->lpPageBuffer[0] = 0;
    }
}

/*******************************************************************************
 *  InitializeOutputBuffer()
 *      Allocate buffers
 *      Those buffers are used to convert strips into device format.
 *      Need to be called at the beginning of the page.
 *
 *  Returns:
 *      None
 *******************************************************************************/
void InitializeOutputBuffer(PDEVDATA pdev, struct settings_ *settings) {
    uint32_t ulChannelOption = pdev->eChannelOption;
    uint32_t ulColour        = 0;

    if ((pdev->lpCMYIn[0] != NULL) && (pdev->lpCMYOut[0] != NULL) &&
        (pdev->lpKOut != NULL)) {
        // All buffer is available
        return;
    }
    TRACE("InitializeOutputBuffer:ulChannelOption: %u", ulChannelOption);
    TRACE("InitializeOutputBuffer:pdev->xImage: %u", pdev->xImage);
    pdev->ulCMYInPlaneWidth = pdev->xImage; // 642

    if ((ulChannelOption == UICBVAL_YMC) || (ulChannelOption == UICBVAL_YMCK)) {
        uint8_t *lpCMYIn  = NULL;
        uint8_t *lpCMYOut = NULL;

        // Create 8 bit and 6 bit plane buffer
        // 2592.. 648 x288
        //        pdev->ulCMYInPlaneWidth = (((pStripObj->Bitmap.ulPlaneWidth /
        //        4) + 7) / 8) * 8;

        // so in  plane width always 0x288 in original ....

        if (RIO_OEM(settings) || AVALON_MODEL(settings)) {
            pdev->ulCMYOutPlaneWidth = OUTPUT_DATA_WIDTH_RIO;
        } else if (HELIX_OEM(settings)) {
            pdev->ulCMYOutPlaneWidth = OUTPUT_DATA_WIDTH_HELIX;
        } else if (PRO360_OEM(settings)) {
            pdev->ulCMYOutPlaneWidth = OUTPUT_DATA_WIDTH_PRO360;
        } else {
            pdev->ulCMYOutPlaneWidth = OUTPUT_DATA_WIDTH_ALTO;
        }

        lpCMYIn  = (uint8_t *)malloc(pdev->ulCMYInPlaneWidth * 3);
        lpCMYOut = (uint8_t *)malloc(pdev->ulCMYOutPlaneWidth * 3);

        if ((lpCMYIn == NULL) || (lpCMYOut == NULL)) {
            free(lpCMYOut);
            free(lpCMYIn);
            goto cleanup_buffer;
        }

        memset(lpCMYIn, 0, pdev->ulCMYInPlaneWidth * 3);
        memset(lpCMYOut, 0, pdev->ulCMYOutPlaneWidth * 3);

        for (ulColour = 0; ulColour < 3; ulColour++) {
            pdev->lpCMYIn[ulColour] =
                    lpCMYIn + ulColour * pdev->ulCMYInPlaneWidth;
            pdev->lpCMYOut[ulColour] =
                    lpCMYOut + ulColour * pdev->ulCMYOutPlaneWidth;
        }
    }

    if ((ulChannelOption == UICBVAL_KResin) ||
        (ulChannelOption == UICBVAL_YMCK)) {
        // Fixed as format_printhead_data() requires this value
        if (RIO_OEM(settings) || AVALON_MODEL(settings)) {
            pdev->ulKOutPlaneWidth = OUTPUT_DATA_WIDTH_RIO / 6;
        } else if (PRO360_OEM(settings)) {
            pdev->ulKOutPlaneWidth = OUTPUT_DATA_WIDTH_PRO360;
        } else if (HELIX_OEM(settings)) {
            pdev->ulKOutPlaneWidth = OUTPUT_DATA_WIDTH_HELIX;
        } else {
            pdev->ulKOutPlaneWidth = OUTPUT_DATA_WIDTH_ALTO / 6;
        }

        pdev->lpKOut = (uint8_t *)malloc(pdev->ulKOutPlaneWidth);
        if (pdev->lpKOut == 0) {
            pdev->eCallBackStatus = DCBS_FATALERROR;
            goto cleanup_buffer;
        }
        memset(pdev->lpKOut, 0, pdev->ulKOutPlaneWidth);
    }

    if ((ulChannelOption == UICBVAL_YMCK) || (ulChannelOption == UICBVAL_YMC)) {
        uint8_t *lpPageBuffer = 0;

        lpPageBuffer = (uint8_t *)malloc(pdev->ulCMYInPlaneWidth *
                                         pdev->yImage * PLANE_MAX);
        if (lpPageBuffer == 0) {
            pdev->eCallBackStatus = DCBS_FATALERROR;
            goto cleanup_buffer;
        }
        TRACE("InitializeOutputBuffer:pdev->yImage: %u", pdev->yImage);
        memset(lpPageBuffer,
               0,
               pdev->ulCMYInPlaneWidth * pdev->yImage * PLANE_MAX);
        for (ulColour = 0; ulColour < PLANE_MAX; ulColour++) { // x282..642 1016
            pdev->lpPageBuffer[ulColour] =
                    lpPageBuffer +
                    ulColour * (pdev->ulCMYInPlaneWidth * pdev->yImage);
        }
    }

cleanup_buffer:
    if (pdev->eCallBackStatus == DCBS_FATALERROR) {
        DeleteOutputBuffer(pdev);
    }

    return;
}

/*****************************************************************************
 *  _initialize_BufferInfo()
 *      Initialize SPOOLMEMINFO structure.
 *      Supply memory alloc function both for user and kernel mode.
 *
 *  Parameters:
 *      lpSplMemInfo    Pointer to SPOOLMEMINFO
 *      ulSize          Required memory size
 *      ulColour        The colour to sent out. It should be one of:
 *                      PLANE_Y, PLANE_M, PLANE_C, PLANE_K
 *
 *  Returns:
 *      None
 *****************************************************************************/
bool _initialize_BufferInfo(LPSPOOLMEMINFO lpSplMemInfo,
                            uint32_t       ulSize,
                            uint32_t       ulColour) {
    /*
     * Call memory allocation function.
     * For Kernel mode we use EngAllocMem and use paged pool.
     */

    lpSplMemInfo->lpBuffer = malloc(ulSize);

    if (lpSplMemInfo->lpBuffer == NULL) {
        return false;
    }

    lpSplMemInfo->ulDataSize   = 0;
    lpSplMemInfo->ulColour     = ulColour;
    lpSplMemInfo->bDataInPlane = false;

    return true;
}

/*****************************************************************************
 *  FreeBuffer()
 *      Clean up memory which were allocated at AllocSpoolMem
 *
 * Parameters:
 *     pdev      Pointer to our PDEV
 *     bFront    true = Front page (or simplex) : false = Back page
 *
 *  Returns:
 *      None
 *****************************************************************************/
void FreeBuffer(PDEVDATA pdev, bool bFront) {
    uint32_t       ulColour = 0;
    LPSPOOLMEMINFO lpSplMem = NULL;

    for (ulColour = 0; ulColour < 4; ulColour++) {
        if (bFront) {
            lpSplMem = &pdev->lpSplMemFront[ulColour];
        } else {
            lpSplMem = &pdev->lpSplMemBack[ulColour];
        }

        if (lpSplMem->lpBuffer != NULL) {
            free(lpSplMem->lpBuffer);
        }

        lpSplMem->lpBuffer     = NULL;
        lpSplMem->ulDataSize   = 0;
        lpSplMem->ulColour     = 0;
        lpSplMem->bDataInPlane = false;
    }
    return;
}

/*****************************************************************************
 *  AllocBuffer()
 *      Alloc memory for page data spool.
 *      In duplex, we spool both front and back pages.
 *
 *  Parameters:
 *      pdev        Pointer to our PDEV
 *      settings
 *      bFront      true = Front page (or simplex) : false = Back page
 *
 *  Returns:
 *      None
 *****************************************************************************/
void AllocBuffer(PDEVDATA pdev, struct settings_ *settings, bool bFront) {
    uint32_t ulMemSize     = 0;
    uint32_t ulMemSizeMono = 0;
    bool     bFailed       = false;

    if (RIO_OEM(settings) || AVALON_MODEL(settings)) {
        ulMemSize     = OUTPUT_DATA_WIDTH_RIO * pdev->yImage;
        ulMemSizeMono = (OUTPUT_DATA_WIDTH_RIO / 6) * pdev->yImage;
    } else if (PRO360_OEM(settings)) {
        ulMemSize = ulMemSizeMono = OUTPUT_DATA_WIDTH_PRO360 * pdev->yImage;
    } else if (HELIX_OEM(settings)) {
        ulMemSize = ulMemSizeMono = OUTPUT_DATA_WIDTH_HELIX * pdev->yImage;
    } else {
        ulMemSize     = OUTPUT_DATA_WIDTH_ALTO * pdev->yImage;
        ulMemSizeMono = (OUTPUT_DATA_WIDTH_ALTO / 6) * pdev->yImage;
    }

    if (XXL_TYPE(settings) && (pdev->eChannelOption == UICBVAL_KResin)) {
        // For extended Monochrome on an XXL type printer, adjust the buffer
        // size
        // for K Resin to be the same as a colour plane
        ulMemSizeMono = ulMemSize;
    }
    TRACE("Alloc %u", bFront);

    if (bFront) {

        switch (pdev->eChannelOption) {
            case UICBVAL_YMC:
                // Switch depends on which side
                bFailed |= !_initialize_BufferInfo(
                        &pdev->lpSplMemFront[PLANE_C], ulMemSize, PLANE_C);
                bFailed |= !_initialize_BufferInfo(
                        &pdev->lpSplMemFront[PLANE_M], ulMemSize, PLANE_M);
                bFailed |= !_initialize_BufferInfo(
                        &pdev->lpSplMemFront[PLANE_Y], ulMemSize, PLANE_Y);
                break;

            case UICBVAL_YMCK:
            default:
                bFailed |= !_initialize_BufferInfo(
                        &pdev->lpSplMemFront[PLANE_C], ulMemSize, PLANE_C);
                bFailed |= !_initialize_BufferInfo(
                        &pdev->lpSplMemFront[PLANE_M], ulMemSize, PLANE_M);
                bFailed |= !_initialize_BufferInfo(
                        &pdev->lpSplMemFront[PLANE_Y], ulMemSize, PLANE_Y);
                bFailed |= !_initialize_BufferInfo(
                        &pdev->lpSplMemFront[PLANE_K], ulMemSizeMono, PLANE_K);
                break;

            case UICBVAL_KResin:
                bFailed |= !_initialize_BufferInfo(
                        &pdev->lpSplMemFront[PLANE_K], ulMemSizeMono, PLANE_K);
                break;
        }
    } else {
        TRACE("eChannelOption %u", pdev->eChannelOption);

        switch (pdev->eChannelOption) {
            case UICBVAL_YMC:
                // Switch depends on which side
                TRACE("eChannelOption %u", pdev->eChannelOption);
                bFailed |= !_initialize_BufferInfo(
                        &pdev->lpSplMemBack[PLANE_C], ulMemSize, PLANE_C);
                bFailed |= !_initialize_BufferInfo(
                        &pdev->lpSplMemBack[PLANE_M], ulMemSize, PLANE_M);
                bFailed |= !_initialize_BufferInfo(
                        &pdev->lpSplMemBack[PLANE_Y], ulMemSize, PLANE_Y);
                bFailed |= !_initialize_BufferInfo(
                        &pdev->lpSplMemBack[PLANE_K], ulMemSizeMono, PLANE_K);

                break;

            case UICBVAL_YMCK:
            default:
              TRACE("eChannelOption %u", pdev->eChannelOption);
                bFailed |= !_initialize_BufferInfo(
                        &pdev->lpSplMemBack[PLANE_C], ulMemSize, PLANE_C);
                bFailed |= !_initialize_BufferInfo(
                        &pdev->lpSplMemBack[PLANE_M], ulMemSize, PLANE_M);
                bFailed |= !_initialize_BufferInfo(
                        &pdev->lpSplMemBack[PLANE_Y], ulMemSize, PLANE_Y);
                bFailed |= !_initialize_BufferInfo(
                        &pdev->lpSplMemBack[PLANE_K], ulMemSizeMono, PLANE_K);
                break;

            case UICBVAL_KResin:
                TRACE("eChannelOption %u", pdev->eChannelOption);
                bFailed |= !_initialize_BufferInfo(
                        &pdev->lpSplMemBack[PLANE_K], ulMemSizeMono, PLANE_K);
                break;
        }
    }

    if (bFailed) {
        TRACE("Alloc %u", bFailed);

        pdev->eCallBackStatus = DCBS_FATALERROR;
        FreeBuffer(pdev, bFront);
    }

    return;
}

/*****************************************************************************
 *  CopyToBuffer()
 *      Copy data into spool buffer.
 *
 *  Parameters:
 *      pdev            Pointer to our PDEV
 *      lpOutBuf        Pointer to plane data
 *      ulOutBufSize    The data length in pPlane
 *      ulColour        The colour to sent out. It should be one
 *                      of: PLANE_Y, PLANE_M, PLANE_C, PLANE_K *
 *      bDataInPlane    true if there is any colour in lpOutBuf
 *
 *  Returns:
 *      None
 *****************************************************************************/
void CopyToBuffer(PDEVDATA pdev,
                  uint8_t *lpOutBuf,
                  uint32_t ulOutBufSize,
                  uint32_t ulColour,
                  bool     bDataInPlane) {
    LPSPOOLMEMINFO lpSplMem = NULL;

    if (!pdev->bFrontPage) {
        lpSplMem = &pdev->lpSplMemBack[ulColour];
    } else {
        lpSplMem = &pdev->lpSplMemFront[ulColour];
    }

    if (bDataInPlane && !lpSplMem->bDataInPlane) {
        // If there is colour we will send out this plane to the device at
        // EndPage.
        lpSplMem->bDataInPlane = true;
    }

    if (lpSplMem->lpBuffer != NULL) {
        memcpy(lpSplMem->lpBuffer + lpSplMem->ulDataSize,
               lpOutBuf,
               ulOutBufSize);
        lpSplMem->ulDataSize += ulOutBufSize;
    }

    return;
}

void GetPixelRGB(PDEVDATA   pdev,
                 uint8_t *  pSurface,
                 PRGBTRIPLE pRGB,
                 uint16_t   x,
                 uint16_t   y) {
    uint8_t *pBits;

    pBits = (uint8_t *)pSurface;
    pBits += (y * (pdev->epPaperXdots * 3));
    pBits += (x * 3);

    pRGB->rgbtRed   = pBits[0];
    pRGB->rgbtGreen = pBits[1];
    pRGB->rgbtBlue  = pBits[2];
}

void SetPixelRGB(
        uint8_t *pSurface, int x, int y, uint8_t r, uint8_t g, uint8_t b) {
    uint8_t *pBits = (uint8_t *)pSurface + (y * (PREVIEW_WIDTH * 3)) + x * 3;

    *pBits++ = b;
    *pBits++ = g;
    *pBits   = r;
}

void Resize(PDEVDATA pdev, uint8_t *pSrc, uint8_t *pDst) {
    int       nWidth  = PREVIEW_WIDTH;
    int       nHeight = PREVIEW_HEIGHT;
    int       x, y;
    double    nXFactor;
    double    nYFactor;
    double    fraction_x, fraction_y, one_minus_x, one_minus_y;
    int       ceil_x, ceil_y, floor_x, floor_y;
    RGBTRIPLE c1;
    RGBTRIPLE c2;
    RGBTRIPLE c3;
    RGBTRIPLE c4;
    uint8_t   red, green, blue;
    uint8_t   b1, b2;

    nXFactor = (double)pdev->epPaperXdots / (double)nWidth;
    nYFactor = (double)pdev->epPaperYdots / (double)nHeight;

    for (y = 0; y < nHeight; y++) {
        for (x = 0; x < nWidth; x++) {
            // Setup

            floor_x = (int)floor(x * nXFactor);
            floor_y = (int)floor(y * nYFactor);
            ceil_x  = floor_x + 1;
            if (ceil_x >= pdev->epPaperXdots) {
                ceil_x = floor_x;
            }
            ceil_y = floor_y + 1;
            if (ceil_y >= pdev->epPaperYdots) {
                ceil_y = floor_y;
            }
            fraction_x  = x * nXFactor - floor_x;
            fraction_y  = y * nYFactor - floor_y;
            one_minus_x = 1.0 - fraction_x;
            one_minus_y = 1.0 - fraction_y;

            GetPixelRGB(pdev, pSrc, (PRGBTRIPLE)&c1, floor_x, floor_y);
            GetPixelRGB(pdev, pSrc, (PRGBTRIPLE)&c2, ceil_x, floor_y);
            GetPixelRGB(pdev, pSrc, (PRGBTRIPLE)&c3, floor_x, ceil_y);
            GetPixelRGB(pdev, pSrc, (PRGBTRIPLE)&c4, ceil_x, ceil_y);

            // Blue
            b1 = (uint8_t)(one_minus_x * c1.rgbtBlue +
                           fraction_x * c2.rgbtBlue);
            b2 = (uint8_t)(one_minus_x * c3.rgbtBlue +
                           fraction_x * c4.rgbtBlue);
            blue = (uint8_t)(one_minus_y * (double)(b1) +
                             fraction_y * (double)(b2));

            // Green
            b1 = (uint8_t)(one_minus_x * c1.rgbtGreen +
                           fraction_x * c2.rgbtGreen);
            b2 = (uint8_t)(one_minus_x * c3.rgbtGreen +
                           fraction_x * c4.rgbtGreen);
            green = (uint8_t)(one_minus_y * (double)(b1) +
                              fraction_y * (double)(b2));

            // Red
            b1  = (uint8_t)(one_minus_x * c1.rgbtRed + fraction_x * c2.rgbtRed);
            b2  = (uint8_t)(one_minus_x * c3.rgbtRed + fraction_x * c4.rgbtRed);
            red = (uint8_t)(one_minus_y * (double)(b1) +
                            fraction_y * (double)(b2));
            // this works..
            SetPixelRGB(pDst, x, y, red, green, blue);
        }
    }
}

uint32_t CMYK_to_RGB(uint8_t C, uint8_t M, uint8_t Y, uint8_t K) {
    uint32_t t;
    float    r, g, b;
    float    c = (float)C;
    float    m = (float)M;
    float    y = (float)Y;
    float    k = (float)K;

    c = MIN(255, c + k);
    m = MIN(255, m + k);
    y = MIN(255, y + k);
    r = 255 - c;
    g = 255 - m;
    b = 255 - y;

    t = (uint8_t)r << 16;
    t += (uint8_t)g << 8;
    t += (uint8_t)b;

    return t;
}

/*****************************************************************************
 *  FlushBuffer()
 *      Send out all data in spool buffer to the device.
 *
 *  Parameters:
 *      pdev         Pointer to our PDEV
 *      settings
 *      bFront       true = Front page (or simplex) : false = Back page
 *
 *  Returns:
 *      None
 *****************************************************************************/
void FlushBuffer(PDEVDATA pdev, struct settings_ *settings, bool bFront) {
    LPSPOOLMEMINFO lpSplMem = 0;
    uint32_t       ulColour = 0;
    bool           bBlank   = true;

    if (settings->OEM == OEM_PRO360) {
        for (ulColour = 0; ulColour < 4; ulColour++) {
            TRACE("ulColour = %d, bFront = %s",
                  ulColour,
                  ShowTrueFalse(bFront));
            if (bFront) {
                lpSplMem = &pdev->lpSplMemFront[ulColour];
            } else {
                lpSplMem = &pdev->lpSplMemBack[ulColour];
            }

            if (lpSplMem->lpBuffer != NULL && lpSplMem->bDataInPlane) {
                TRACE("Calling OutputPlaneCommand() ulColour = %d, "
                      "lpSplMem->ulDataSize = %d",
                      ulColour,
                      lpSplMem->ulDataSize);
                OutputPlaneCommand(pdev,
                                   settings,
                                   lpSplMem->lpBuffer,
                                   lpSplMem->ulColour,
                                   (settings->xdpi == 600 && ulColour != 3) ?
                                           lpSplMem->ulDataSize >> 1 :
                                           lpSplMem->ulDataSize);
                bBlank = false;
            }
            TRACE_AT;
        }

        TRACE_AT;

        // we have a blank card so print a blank card!
        if (bBlank &&
            (pdev->magTrackInfo[MAG_TRACK_INDEX1].TrackData[0] == 0) &&
            (pdev->magTrackInfo[MAG_TRACK_INDEX2].TrackData[0] == 0) &&
            (pdev->magTrackInfo[MAG_TRACK_INDEX3].TrackData[0] == 0)) {
            if (bFront) {
                lpSplMem = &pdev->lpSplMemFront[PLANE_K];
            } else {
                lpSplMem = &pdev->lpSplMemBack[PLANE_K];
            }

            TRACE_STR("Calling OutputPlaneCommand() with K plane to print a "
                      "blank card.");

            OutputPlaneCommand(pdev,
                               settings,
                               lpSplMem->lpBuffer,
                               lpSplMem->ulColour,
                               pdev->ulKOutPlaneWidth *
                                       1013 /*lpSplMem->ulDataSize*/);
            TRACE_AT;
        }
        TRACE_AT;
    } // ends if pro360
    else if (settings->OEM == OEM_HELIX) {
        for (ulColour = 0; ulColour < 4; ulColour++) {
            if (bFront) {
                lpSplMem = &pdev->lpSplMemFront[ulColour];
            } else {
                lpSplMem = &pdev->lpSplMemBack[ulColour];
            }

            if (lpSplMem->lpBuffer != NULL && lpSplMem->bDataInPlane) {
                OutputPlaneCommand(pdev,
                                   settings,
                                   lpSplMem->lpBuffer,
                                   lpSplMem->ulColour,
                                   lpSplMem->ulDataSize);
                bBlank = false;
            }
        }

        // we have a blank card so print a blank card!
        if (bBlank &&
            (pdev->magTrackInfo[MAG_TRACK_INDEX1].TrackData[0] == 0) &&
            (pdev->magTrackInfo[MAG_TRACK_INDEX2].TrackData[0] == 0) &&
            (pdev->magTrackInfo[MAG_TRACK_INDEX3].TrackData[0] == 0)) {
            if (bFront) {
                lpSplMem = &pdev->lpSplMemFront[PLANE_K];
            } else {
                lpSplMem = &pdev->lpSplMemBack[PLANE_K];
            }

            OutputPlaneCommand(pdev,
                               settings,
                               lpSplMem->lpBuffer,
                               lpSplMem->ulColour,
                               lpSplMem->ulDataSize);
        }

        // generate our preview data
        // now throw out the preview data for the side
        // generated from 32bit CMYK data

        if (pdev->bEncodeOnlyJob == false) {

            uint8_t *pPreviewSurface = 0;
            uint8_t *pOutputSurface  = 0;
            int      rgb888, row, col;
            uint16_t x, y, x2, y2;

            pPreviewSurface = malloc((PREVIEW_WIDTH * PREVIEW_HEIGHT) * 3);
            pOutputSurface =
                    malloc((pdev->epPaperXdots * pdev->epPaperYdots) * 3);

            if (pPreviewSurface) {
                uint8_t *lpSrc = NULL;
                uint8_t *lpDst = NULL;

                LPSPOOLMEMINFO lpSplMemY = NULL;
                LPSPOOLMEMINFO lpSplMemM = NULL;
                LPSPOOLMEMINFO lpSplMemC = NULL;
                LPSPOOLMEMINFO lpSplMemK = NULL;

                int      yVal = 0, mVal = 0, cVal = 0, kVal = 0;
                uint32_t rgbval;

                // generate preview here :)

                if (bFront) {
                    lpSplMemY = &pdev->lpSplMemFront[0];
                    lpSplMemM = &pdev->lpSplMemFront[1];
                    lpSplMemC = &pdev->lpSplMemFront[2];
                    lpSplMemK = &pdev->lpSplMemFront[3];

                    for (y = 0; y < pdev->epPaperYdots; y++) {
                        for (x = 0; x < pdev->epPaperXdots; x++) {
                            if (lpSplMemY->ulDataSize) {
                                yVal = lpSplMemY->lpBuffer
                                               [(y * OUTPUT_DATA_WIDTH_HELIX) +
                                                x];
                            }
                            if (lpSplMemM->ulDataSize) {
                                mVal = lpSplMemM->lpBuffer
                                               [(y * OUTPUT_DATA_WIDTH_HELIX) +
                                                x];
                            }
                            if (lpSplMemC->ulDataSize) {
                                cVal = lpSplMemC->lpBuffer
                                               [(y * OUTPUT_DATA_WIDTH_HELIX) +
                                                x];
                            }
                            if (lpSplMemK->ulDataSize) {
                                kVal = lpSplMemK->lpBuffer
                                               [(y * OUTPUT_DATA_WIDTH_HELIX) +
                                                x];
                            }

                            rgbval = CMYK_to_RGB((uint8_t)cVal,
                                                 (uint8_t)mVal,
                                                 (uint8_t)yVal,
                                                 (uint8_t)kVal);
                            lpSrc = (uint8_t *)pOutputSurface;
                            lpSrc += (y * (pdev->epPaperXdots * 3));
                            lpSrc += (x * 3);
                            *lpSrc++ = GetRValue(rgbval);
                            *lpSrc++ = GetGValue(rgbval);
                            *lpSrc   = GetBValue(rgbval);
                        }
                    }
                } else {
                    lpSplMemY = &pdev->lpSplMemBack[0];
                    lpSplMemM = &pdev->lpSplMemBack[1];
                    lpSplMemC = &pdev->lpSplMemBack[2];
                    lpSplMemK = &pdev->lpSplMemBack[3];

                    for (y2 = 0, y = pdev->epPaperYdots - 1; y; y--, y2++) {
                        for (x2 = 0, x = pdev->epPaperXdots - 3; x; x--, x2++) {
                            if (lpSplMemY->ulDataSize) {
                                yVal = lpSplMemY->lpBuffer
                                               [(y * OUTPUT_DATA_WIDTH_HELIX) +
                                                x];
                            }
                            if (lpSplMemM->ulDataSize) {
                                mVal = lpSplMemM->lpBuffer
                                               [(y * OUTPUT_DATA_WIDTH_HELIX) +
                                                x];
                            }
                            if (lpSplMemC->ulDataSize) {
                                cVal = lpSplMemC->lpBuffer
                                               [(y * OUTPUT_DATA_WIDTH_HELIX) +
                                                x];
                            }
                            if (lpSplMemK->ulDataSize) {
                                kVal = lpSplMemK->lpBuffer
                                               [(y * OUTPUT_DATA_WIDTH_HELIX) +
                                                x];
                            }

                            rgbval = CMYK_to_RGB((uint8_t)yVal,
                                                 (uint8_t)mVal,
                                                 (uint8_t)cVal,
                                                 (uint8_t)kVal);
                            lpSrc = (uint8_t *)pOutputSurface;
                            lpSrc += (y2 * (pdev->epPaperXdots * 3));
                            lpSrc += x2 * 3;
                            *lpSrc++ = GetRValue(rgbval);
                            *lpSrc++ = GetGValue(rgbval);
                            *lpSrc   = GetBValue(rgbval);
                        }
                    }
                }

                Resize(pdev, pOutputSurface, pPreviewSurface);

                for (row = 0; row < PREVIEW_HEIGHT; row++) {
                    for (col = 0; col < PREVIEW_WIDTH; col++) {
                        lpDst = (uint8_t *)pPreviewSurface +
                                (row * (PREVIEW_WIDTH * 3)) + (col * 3);
                        rgb888 = (uint32_t)RGB(
                                *lpDst, *(lpDst + 1), *(lpDst + 2));
                        pdev->HelixPreview[(PREVIEW_HEIGHT - 1) - row][col] =
                                (uint16_t)RGB_565(rgb888);
                    }
                }
                // discard our created surfaces
                free(pPreviewSurface);
                free(pOutputSurface);
            }

            // Send the preview data
            OutputPlaneCommand(pdev,
                               settings,
                               (uint8_t *)&pdev->HelixPreview,
                               0,
                               (PREVIEW_HEIGHT * PREVIEW_WIDTH) * 2);
        }

        // ends if helix
    } else {

        for (ulColour = 0; ulColour < 4; ulColour++) {
            if (bFront) {
                lpSplMem = &pdev->lpSplMemFront[ulColour];
            } else {
                lpSplMem = &pdev->lpSplMemBack[ulColour];
            }

            if (lpSplMem->lpBuffer != NULL && lpSplMem->bDataInPlane) {
                OutputPlaneCommand(pdev,
                                   settings,
                                   lpSplMem->lpBuffer,
                                   lpSplMem->ulColour,
                                   lpSplMem->ulDataSize);
                bBlank = false;
            }
        }

        // we have a blank card so print a blank card! removed due to firmware
        // not feeding a card <groan>
        if (bBlank &&
            (pdev->magTrackInfo[MAG_TRACK_INDEX1].TrackData[0] == 0) &&
            (pdev->magTrackInfo[MAG_TRACK_INDEX2].TrackData[0] == 0) &&
            (pdev->magTrackInfo[MAG_TRACK_INDEX3].TrackData[0] == 0)) {
            if (bFront) {
                lpSplMem = &pdev->lpSplMemFront[PLANE_K];
            } else {
                lpSplMem = &pdev->lpSplMemBack[PLANE_K];
            }

            OutputPlaneCommand(pdev,
                               settings,
                               lpSplMem->lpBuffer,
                               lpSplMem->ulColour,
                               97536 /*lpSplMem->ulDataSize*/);
        }

        OutputETXCommand();
    }
}

/*****************************************************************************
 *  SetDataSize()
 *      Send SZX commands.
 *      If we have not seen any colour in a plane, the command won't be out.
 *
 *  Parameters:
 *        pdev     Pointer to our PDEV
 *        bFront   true = Front page (or simplex) : false = Back page
 *
 *  Returns:
 *      None
 *****************************************************************************/
void SetDataSize(PDEVDATA pdev, bool bFront) {
    LPSPOOLMEMINFO lpSplMem   = NULL;
    uint32_t       ulColour   = 0;
    uint8_t        ColourMask = 0;
    bool           bBlank     = true;

    ColourMask = GetDataPresent(pdev, bFront);

    for (ulColour = 0; ulColour < 4; ulColour++) {
        if (bFront) {
            lpSplMem = &pdev->lpSplMemFront[ulColour];
        } else {
            lpSplMem = &pdev->lpSplMemBack[ulColour];
        }

        if (ColourMask & (1 << ulColour)) {
            OutputPlaneDataSizeCommand(lpSplMem->ulDataSize,
                                       lpSplMem->ulColour);

            // Clear bit for this colour
            ColourMask &= ~(1 << ulColour);
            if (ColourMask != 0) {
                // There are more colours to handle so output a comma
                OutputComma();
            }
            bBlank = false;
        }
    }

    // removed due to firmware not feeding a card <groan>
    if (bBlank && (pdev->magTrackInfo[MAG_TRACK_INDEX1].TrackData[0] == 0) &&
        (pdev->magTrackInfo[MAG_TRACK_INDEX2].TrackData[0] == 0) &&
        (pdev->magTrackInfo[MAG_TRACK_INDEX3].TrackData[0] == 0)) {
        // need to determine which plane is available on the printer CMY or K
        // resin installed?
        OutputPlaneDataSizeCommand(97536, PLANE_K);
    }
    OutputFileSeparator();
}

/*******************************************************************************
 *  NewPage()
 *      Initialize information per page.
 *
 *  Returns:
 *      True/False
 *******************************************************************************/

bool NewPage(PDEVDATA            pdev,
             struct settings_ *  settings,
             cups_page_header2_t header) {
    int iCurrentPage = pdev->iPageNumber;

    GetDuplexInfo(pdev, settings, iCurrentPage);

    if (pdev->eChannelOption == UICBVAL_YMCK) {
        if (pdev->bFrontPage) {
            pdev->bBlackTextUseK =
                    settings->CardFront.BlackOptions.BlackTextUsesKResin;
            pdev->bBlackPolygonUseK =
                    settings->CardFront.BlackOptions.BlackPolygonsUseKResin;
            pdev->bMonoBitmapUseK =
                    settings->CardFront.BlackOptions.MonoBitmapsUseKResin;
            pdev->bPhotoUseYMC =
                    settings->CardFront.BlackOptions.PicturesUseYMConly;
            pdev->bDetectAdjacentColour = true;
        } else {
            pdev->bBlackTextUseK =
                    settings->CardBack.BlackOptions.BlackTextUsesKResin;
            pdev->bBlackPolygonUseK =
                    settings->CardBack.BlackOptions.BlackPolygonsUseKResin;
            pdev->bMonoBitmapUseK =
                    settings->CardBack.BlackOptions.MonoBitmapsUseKResin;
            pdev->bPhotoUseYMC =
                    settings->CardBack.BlackOptions.PicturesUseYMConly;
            pdev->bDetectAdjacentColour = true;
        }
    }

    if ((pdev->eChannelOption == UICBVAL_YMCK) ||
        (pdev->eChannelOption == UICBVAL_KResin)) {
        // Create Strip for KObject
        if (CreateKPrintingObject(pdev, settings, header) == false) {
            // Failed to alloc buffer
            return false;
        }
    }

    InitializeOutputBuffer(pdev, settings);

    if (pdev->eCallBackStatus == DCBS_FATALERROR) {
        // Failed to alloc buffer
        return false;
    }

    AllocBuffer(pdev, settings, pdev->bFrontPage);

    if (pdev->eCallBackStatus == DCBS_FATALERROR) {
        // Failed to alloc buffer
        return false;
    }

    pdev->bPageStarted = true;

    return true;
}

void pageSetup(PDEVDATA            pdev,
               struct settings_    settings,
               cups_page_header2_t header) {
    uint32_t          x;
    bool              bBackPage = false;
    struct settings_ *psettings = &settings;
    TRACE_IN;

    TRACE("settings.Duplex = %d", settings.Duplex);
    TRACE("settings.pageType = %d", settings.pageType);
    TRACE("settings.OEM = %d", settings.OEM);
    TRACE("settings.Printer = %d", settings.Printer);

    if (settings.OEM == OEM_PRO360) {
        TRACE_STR("Pro360 family - pageSetup");

        // Set the default pts for a CR80 card
        pdev->xImage = 642;
        pdev->width_pts = 1013;

        if (PRO360_XTD_TYPE(psettings)) {
           TRACE_STR("Setting up for an eXtended printer.");
           TRACE("XXL_ImageType = %d", psettings->XXL_ImageType);

            if (psettings->XXL_ImageType == UICBVAL_DoubleImage ||
                  psettings->XXL_ImageType == UICBVAL_Extended_Monochrome ||
                  psettings->XXL_ImageType == UICBVAL_XLTape) {

               // Set the yImage variable to be the length of the card, in
               // 300ths of an inch (pts).
                if (psettings->pageType == 3) {
                    // 140mm
                    TRACE_STR("Selected card size is 140mm");
                    pdev->width_pts = 1654;
                } else if (psettings->pageType == 4  /* Extended Mono */ ||
                           psettings->pageType == 2 /* Extended Colour */ ) {
                    // 109mm
                    TRACE_STR("Selected card size is 109mm");
                    pdev->width_pts = 1287;
                } else if (psettings->pageType == 5) {
                    // 128mm
                    TRACE_STR("Selected card size is 128mm");
                    pdev->width_pts = 1512;
                }
            }
        } // End if PRO360_XTD_TYPE
    } else if (psettings->OEM == OEM_HELIX) {
        pdev->xImage = 1036;
        pdev->yImage = 664;
    } else {
        pdev->xImage = 642;
        pdev->yImage = 1016;
    }

    pdev->epOffsetX = 0;
    pdev->epOffsetY = 0;

    if (settings.OEM == OEM_PRO360) {
        // init the pro360 job crc value
        pdev->epCheckSum = 0xffffffffu;
        if (settings.xdpi == 600) {
            pdev->epPhysPaperXdots = 1284;
            pdev->epPhysPaperYdots = 2026;
            pdev->epPaperXdots = 1284; // set to pro360 panel size
            pdev->epPaperYdots = 2026;
            pdev->yImage       = pdev->width_pts * 2;
        } else {
            pdev->epPhysPaperXdots = 642;
            pdev->epPhysPaperYdots = header.cupsHeight;
            pdev->epPaperXdots = 642; // set to pro360 panel size
            pdev->epPaperYdots = pdev->width_pts;//header.cupsHeight;
            pdev->yImage       = pdev->width_pts;
        }
    } else if (settings.OEM == OEM_HELIX) {
        pdev->epPhysPaperXdots = 1036;
        pdev->epPhysPaperYdots = 664;
        pdev->epPaperXdots     = 1036; // set to optima panel size
        pdev->epPaperYdots     = 664;

    } else if ((settings.Printer == XXL) &&
               settings.XXL_ImageType > UICBVAL_DoubleImage) {
        // handle extended monochrome and tape for xxl
        float fcx = (settings.PaperHeight * (float)300) / (float)254;
        int   icx = (int)(fcx + 0.5f);
        pdev->epPhysPaperXdots = pdev->epPaperXdots = (short)icx;
        pdev->epPhysPaperYdots = pdev->epPaperYdots = 642; // fixed for now..
    } else if (settings.Printer == RIO_PRO_XTENDED ||
               settings.Printer == PRIDENTOPROXTD ||
               settings.Printer == PRICECARD_PRO_FLEX) {
        TRACE("settings.XXL_ImageType = %d", settings.XXL_ImageType);
        TRACE("UICBVAL_DoubleImage = %d", UICBVAL_DoubleImage);
        if (settings.Printer == PRICECARD_PRO_FLEX) {
            // different index in PPD :(
            if (settings.XXL_ImageType == UICBVAL_DuplexImage ||
                  settings.XXL_ImageType == UICBVAL_Extended_Monochrome ||
                  settings.XXL_ImageType == UICBVAL_XLTape) {
                float fcx = (settings.PaperHeight * (float)300) / (float)254;
                int   icx = (int)(fcx + 0.5f);
                pdev->epPhysPaperXdots = pdev->epPaperXdots = (short)icx;
                pdev->epPhysPaperYdots = pdev->epPaperYdots = 642;
                pdev->xImage                                = 642;
                if (psettings->pageType == 1) {
                    // extd mono card 140mm
                    pdev->yImage = 1654;
                } else if (psettings->pageType == 3) {
                    // extd mono card 109mm
                    pdev->yImage = 1287;
                } else if (psettings->pageType == 4) {
                    // extd mono card 128mm
                    pdev->yImage = 1512;
                }
                TRACE("settings.PaperHeight = %d", settings.PaperHeight);
                TRACE("icx = %d", icx);
            } else {
                pdev->epPhysPaperXdots = 1016;
                pdev->epPhysPaperYdots = 642;
                //
                // set to cr80 size
                pdev->epPaperXdots = 1016;
                pdev->epPaperYdots = 642;
                TRACE("settings.PaperHeight = %d", settings.PaperHeight);
            }
        } else {
            if (settings.XXL_ImageType > UICBVAL_DoubleImage) {
                float fcx = (settings.PaperHeight * (float)300) / (float)254;
                int   icx = (int)(fcx + 0.5f);
                pdev->epPhysPaperXdots = pdev->epPaperXdots = (short)icx;
                pdev->epPhysPaperYdots = pdev->epPaperYdots = 642;
                pdev->xImage                                = 642;
                if (psettings->pageType == 3) {
                    // extd mono card 140mm
                    pdev->yImage = 1654;
                } else if (psettings->pageType == 4) {
                    // extd mono card 109mm
                    pdev->yImage = 1287;
                } else if (psettings->pageType == 5) {
                    // extd mono card 128mm
                    pdev->yImage = 1512;
                }
                TRACE("settings.PaperHeight = %d", settings.PaperHeight);
                TRACE("icx = %d", icx);
            } else {
                pdev->epPhysPaperXdots = 1016;
                pdev->epPhysPaperYdots = 642;

                // set to cr80 size
                pdev->epPaperXdots = 1016;
                pdev->epPaperYdots = 642;
                TRACE("settings.PaperHeight = %d", settings.PaperHeight);
            }
        }
    } else {

        if (settings.pageType == 3) {
            // cr80 white border
            pdev->epPhysPaperXdots = 1016;
            pdev->epPhysPaperYdots = 642;
            pdev->epPaperXdots     = 1016;
            pdev->epPaperYdots     = 642;
            pdev->epOffsetX        = 0;
            pdev->epOffsetY        = 23;

        } else {
            pdev->epPhysPaperXdots = 1016;
            pdev->epPhysPaperYdots = 642;

            // set to cr80 size
            pdev->epPaperXdots = 1016;
            pdev->epPaperYdots = 642;
        }
    }

    // paper height required for xtended models
    pdev->iPaperHeight = settings.PaperHeight;

    /*
     * Determine whether we are printing on the front or back of a card.
     * This depends on both the Duplex setting and physical page number.
     */
    if (settings.Duplex == UICBVAL_Duplex_BothSides) {
        if ((pdev->iPageNumber % 2) == 0) {
            // Even page
            bBackPage = true;
        } else {
            // Odd page
            bBackPage = false;
        }
    } else if (settings.Duplex == UICBVAL_Duplex_Back) {
        bBackPage = true;
    } else {
        // lpds->eDuplex is UICBVAL_Duplex_Front
        bBackPage = false;
    }

    /*
     * Record various UICBs for use in evaluating driver dependencies when we
     * call ModelUICBtoDM() below.
     * These depend on whether we are currently printing a back or front page.
     */
    if (bBackPage) {
        pdev->eChannelOption    = GetChannelOption(&settings, BACK);
        pdev->eBlackStartOption = settings.CardBack.BlackOptions.AllBlackAs;
        pdev->eOrientation      = settings.CardBack.CardOrient;
        pdev->eHalftoning       = settings.CardBack.BlackOptions.Halftone;
        pdev->NoBlackAreas      = 0;
        if (settings.CardBack.NoBlackAreas < MAX_BLACK_AREAS) {
            for (x = 0; x < settings.CardBack.NoBlackAreas; x++) {
                pdev->BlackAreas[x].left = settings.CardBack.BlackAreas[x].left;
                pdev->BlackAreas[x].top =
                        settings.CardBack.BlackAreas[x].bottom;
                pdev->BlackAreas[x].right =
                        settings.CardBack.BlackAreas[x].left +
                        settings.CardBack.BlackAreas[x].width;
                pdev->BlackAreas[x].bottom =
                        settings.CardBack.BlackAreas[x].bottom +
                        settings.CardBack.BlackAreas[x].height;
                pdev->NoBlackAreas++;
            }
        }
    } else {
       // Front Page
        pdev->eChannelOption    = GetChannelOption(&settings, FRONT);
        pdev->eBlackStartOption = settings.CardFront.BlackOptions.AllBlackAs;
        pdev->eOrientation      = settings.CardFront.CardOrient;
        pdev->eHalftoning       = settings.CardFront.BlackOptions.Halftone;
        pdev->NoBlackAreas      = 0;

        if (settings.CardFront.NoBlackAreas < MAX_BLACK_AREAS) {
            for (x = 0; x < settings.CardFront.NoBlackAreas; x++) {
                if (settings.CardFront.BlackAreas[x].left == 0 &&
                    settings.CardFront.BlackAreas[x].bottom == 0 &&
                    settings.CardFront.BlackAreas[x].width == 0 &&
                    settings.CardFront.BlackAreas[x].height == 0) {
                    continue;
                }

                pdev->BlackAreas[x].left =
                        settings.CardFront.BlackAreas[x].left;
                pdev->BlackAreas[x].top =
                        642 - settings.CardFront.BlackAreas[x].bottom -
                        settings.CardFront.BlackAreas[x].height;
                pdev->BlackAreas[x].right =
                        settings.CardFront.BlackAreas[x].left +
                        settings.CardFront.BlackAreas[x].width;
                pdev->BlackAreas[x].bottom =
                        642 - settings.CardFront.BlackAreas[x].bottom;
                pdev->NoBlackAreas++;
            }
        }
    }

    /*
     * bImageClipped becomes true (below) if the orientation differs between
     * the application and the driver.
     */
    pdev->bImageClipped = false;
    if (settings.orientation == 0) {
        pdev->iOrientation = UICBVAL_Portrait;
    } else {
        pdev->iOrientation = UICBVAL_Landscape;
    }

    // Record rotate setting
    if (bBackPage) {
        if (pdev->iOrientation == UICBVAL_Portrait) {
            /*
             * Please note that with portrait printing on the back side, the
             * device flips the card...
             * Logical page: No rotate  -> Physical result 180 degrees rotation
             * against the first page
             * Logical page: 180 rotate -> Physical result no rotation against
             * the first page
             */
            pdev->bRotate = !settings.CardBack.Rotate180;
        } else {
            // Landscape on the back side
            pdev->bRotate = settings.CardBack.Rotate180;
        }
    } else {
        // Front page
        pdev->bRotate = settings.CardFront.Rotate180;
    }

    // we default to a TOPDOWN surface which is the opposite of the existing
    if (RIO_OEM(psettings)) {
        pdev->bRotate = !pdev->bRotate;
    } else if (ENDURO_OEM(psettings)) {
        if (RIO_PRO_X_MODEL(psettings)) {
            // Its a Rio Pro Xtended -
            // If image is Extended Colour Card:
            //	Landscape: Do not rotate either front or back
            //	Portrait:  Rotate the front page
            if ((settings.XXL_ImageType == UICBVAL_DoubleImage) &&
                (settings.orientation == 0 /*DMORIENT_PORTRAIT*/) &&
                (bBackPage == false)) {
                // Rotate the front page
                pdev->bRotate = pdev->bRotate;
            } else
                pdev->bRotate = !pdev->bRotate;
        }
        pdev->bRotate = !pdev->bRotate;
    }

    // create our 32bit surface
    TRACE("pdev->bRotate:%u pdev->xImage: %u", pdev->bRotate, pdev->xImage);
    TRACE("CATULTRA pdev->yImage: %u", pdev->yImage);
    pdev->pso32bitSurface = calloc((header.cupsBytesPerLine * header.cupsHeight) * 2 /* sides */, 1);

    if (NewPage(pdev, psettings, header) == false) {
        // DBGMSG(DBG_LEVEL_VERBOSE,"Error creating surface\n");
        return;
    }
}

/*******************************************************************************
 *  DeleteKPrintingObject()
 *      Delete objects which were created at CreateKPrintingObject(),
 *
 *  Returns:
 *      None
 *******************************************************************************/
void DeleteKPrintingObject(PDEVDATA pdev) {

    // 8bpp...
    if (pdev->lpKSrcStrip != NULL) {
        free(pdev->lpKSrcStrip);
        pdev->lpKSrcStrip = NULL;
    }

    if (pdev->lpKPSrcStrip != NULL) {
        free(pdev->lpKPSrcStrip);
        pdev->lpKPSrcStrip = NULL;
    }

    if (pdev->lpKDstStrip != NULL) {
        free(pdev->lpKDstStrip);
        pdev->lpKDstStrip = NULL;
    }

    // 1bpp
    if (pdev->lpHalftone != NULL) {
        free(pdev->lpHalftone);
        pdev->lpHalftone = NULL;
    }
}

#define BITS_FOR_COLOUR_DATA 6
#define BITS_FOR_BLACK_DATA 1
#define DEFAULT_PRINTHEAD_POSN 50
#define PRINTHEAD_WIDTH 672

// Full Bleed CR80 Card Sizing
#define FULL_BLEED_WIDTH 642
#define FULL_BLEED_MARGIN (PRINTHEAD_WIDTH - FULL_BLEED_WIDTH) / 2
#define FULL_BLEED_MAX_MARGIN (DEFAULT_PRINTHEAD_POSN + FULL_BLEED_MARGIN)
#define FULL_BLEED_MIN_MARGIN (DEFAULT_PRINTHEAD_POSN - FULL_BLEED_MARGIN)

// White Border CR80 Card Sizing
#define BORDER_CARD_WIDTH 596
#define BORDER_CARD_MARGIN (PRINTHEAD_WIDTH - BORDER_CARD_WIDTH) / 2
#define BORDER_CARD_MAX_MARGIN (DEFAULT_PRINTHEAD_POSN + BORDER_CARD_MARGIN)
#define BORDER_CARD_MIN_MARGIN (DEFAULT_PRINTHEAD_POSN - BORDER_CARD_MARGIN)

#define MARGIN(x) (PRINTHEAD_WIDTH - (x)) / 2
#define MAX_MARGIN(x) (DEFAULT_PRINTHEAD_POSN + MARGIN((x)))
#define MIN_MARGIN(x) (DEFAULT_PRINTHEAD_POSN - MARGIN((x)))
#define RJE_FULL_BLEED_MAX_MARGIN MAX_MARGIN(FULL_BLEED_WIDTH)
#define RJE_FULL_BLEED_MIN_MARGIN MIN_MARGIN(FULL_BLEED_WIDTH)
#define RJE_WHITE_BORDER_MAX_MARGIN MAX_MARGIN(BORDER_CARD_WIDTH)
#define RJE_WHITE_BORDER_MIN_MARGIN MIN_MARGIN(BORDER_CARD_WIDTH)

//#############################################################################

uint32_t GetPrintheadPosition(PSETTINGS pSettings) {
    uint32_t dwPrintheadPosition = DEFAULT_PRINTHEAD_POSITION;

    if (pSettings->es.iPrintHeadPosn) {
        dwPrintheadPosition = pSettings->es.iPrintHeadPosn;
    }
    return dwPrintheadPosition;
}

//#############################################################################
//##                                                                         ##
//##                              RIO2 PRINTHEAD                             ##
//##                                                                         ##
//#############################################################################

/******************************************************************************
 *  format_printhead_data_KEE()
 *      Processes data line by line and places it into the Magicard Rio/Tango 2e
 *      or Avalon Printhead Format.
 *
 *  Parameters:
 *      psettings    Pointer to the settings struct
 *      lpData       Input Data Buffer
 *      lpOutBuff    Output Data Buffer
 *      nNoOfBits    Colour or Monochrome data
 *      nSize        Size of dOutput Data Buffer

 *  Returns:
 *     None
 ******************************************************************************/
void format_printhead_data_KEE(PSETTINGS psettings,
                               uint8_t * lpData,
                               uint8_t * lpOutBuff,
                               int       nNoOfBits,
                               int       nSize) {
    int       pixel        = 0;
    int       y_index      = 0;
    int       line_length  = 0;
    int       print_bit    = 0;
    int       margin_index = 0;
    int       a            = 0;
    uint32_t  density      = 0;
    uint32_t *temp         = 0;
    uint32_t  dwtemp       = 0;
    uint32_t  temp_word[BITS_FOR_COLOUR_DATA];

    // registry setting aquired from language monitor value returned..
    // other value from UI..range 15 to -15...
    int nPrintheadPosition =
            GetPrintheadPosition(psettings) - psettings->nImagePosition_UpDown;

    memset(lpOutBuff, 0, nSize);

    if (psettings->pageType == 3 /*DMPAPER_WHITEBORDER*/) {
        nPrintheadPosition = MIN(nPrintheadPosition, BORDER_CARD_MAX_MARGIN);
        nPrintheadPosition = MAX(nPrintheadPosition, BORDER_CARD_MIN_MARGIN);
        line_length        = BORDER_CARD_WIDTH;
        margin_index       = BORDER_CARD_MAX_MARGIN - nPrintheadPosition;
    } else {
        nPrintheadPosition = MIN(nPrintheadPosition, FULL_BLEED_MAX_MARGIN);
        nPrintheadPosition = MAX(nPrintheadPosition, FULL_BLEED_MIN_MARGIN);
        line_length        = FULL_BLEED_WIDTH;
        margin_index       = FULL_BLEED_MAX_MARGIN - nPrintheadPosition;
    }

    temp = (uint32_t *)(lpOutBuff);

    if (nNoOfBits > 1) {
        while (y_index < PRINTHEAD_WIDTH) {
            memset((uint8_t *)temp_word, 0, sizeof(temp_word));
            for (print_bit = 0; print_bit < 32; print_bit++) {
                if ((y_index++ > margin_index) && (pixel < line_length)) {
                    pixel++;
                    density = *lpData++;
                    temp_word[0] |= ((density & 0x01) << print_bit);
                    density >>= 1;
                    temp_word[1] |= ((density & 0x01) << print_bit);
                    density >>= 1;
                    temp_word[2] |= ((density & 0x01) << print_bit);
                    density >>= 1;
                    temp_word[3] |= ((density & 0x01) << print_bit);
                    density >>= 1;
                    temp_word[4] |= ((density & 0x01) << print_bit);
                    density >>= 1;
                    temp_word[5] |= ((density & 0x01) << print_bit);
                }
            }

            *temp |= temp_word[0];
            temp += 21;
            *temp |= temp_word[1];
            temp += 21;
            *temp |= temp_word[2];
            temp += 21;
            *temp |= temp_word[3];
            temp += 21;
            *temp |= temp_word[4];
            temp += 21;
            *temp |= temp_word[5];
            temp -= 104; // - (5 * 21) + 1
        }                // end while ( pixel < PRINTHEAD_WIDTH )
    } else {
        uint8_t bSrcBit = 0x80;

        while (y_index < PRINTHEAD_WIDTH) {
            temp_word[0] = 0;
            for (print_bit = 0; print_bit < 32; print_bit++) {
                if (((y_index++) > margin_index) && (pixel < line_length)) {
                    if (*lpData & bSrcBit) {
                        temp_word[0] |= 0x01 << print_bit;
                    }

                    if (bSrcBit == 0x01) {
                        lpData++;
                        bSrcBit = 0x80;
                    } else {
                        bSrcBit >>= 1;
                    }
                    pixel++;
                }
            }
            *temp |= temp_word[0];
            temp++;
        }
    }

    // Re-order bytes/bits
    if (AVALON_MODEL(psettings)) {
        /*
         * uint32_t order will be changed here
         * (7-0) (15-8) (23-16) (31-24)
         */
        for (a = 0; a < (nNoOfBits * 21); a++) {
            dwtemp = 0;
            temp   = (uint32_t *)(lpOutBuff + a * 4);
            dwtemp |= *temp >> 24;
            dwtemp |= (*temp >> 8) & 0x0000FF00;

            dwtemp |= (*temp << 8) & 0x00FF0000;
            dwtemp |= (*temp << 24) & 0xFF000000;
            *temp = dwtemp;
        }
    } else {
        for (a = 0; a < (nNoOfBits * 21); a++) {
            temp  = (uint32_t *)(lpOutBuff + (a * 4));
            *temp = (((*temp & 0xaaaaaaaa) >> 1) | ((*temp & 0x55555555) << 1));
            *temp = (((*temp & 0xcccccccc) >> 2) | ((*temp & 0x33333333) << 2));
            *temp = (((*temp & 0xf0f0f0f0) >> 4) | ((*temp & 0x0f0f0f0f) << 4));
        }
    }
}

//#############################################################################
//##                                                                         ##
//##                              ALTO PRINTHEAD                             ##
//##                                                                         ##
//#############################################################################

#define PIXELS_PART_1 288

typedef struct {
    int first_bit;
    int first_word;
} PIXEL_LAYOUT;

// This table defines the pixel organisation for the data format for the KGE
// printhead
// For pixels 1 to 288, the table is defined as:
//      print_format[a].first_bit  = (a % 16) * 2;
//      print_format[a].first_word = (a / 16) + 6;
// For pixels 289 to 672 the table is defined as
//      print_format[a].first_bit  = ((a - 288) % 16) * 2 + 1;
//      print_format[a].first_word =  (a - 288)/ 16;

const PIXEL_LAYOUT print_format[PRINTHEAD_WIDTH] = {
        {0, 6},   {2, 6},   {4, 6},   {6, 6},
        {8, 6},   {10, 6},  {12, 6},  {14, 6}, // Pixels   0 -   7
        {16, 6},  {18, 6},  {20, 6},  {22, 6},
        {24, 6},  {26, 6},  {28, 6},  {30, 6}, // Pixels   8 -  15
        {0, 7},   {2, 7},   {4, 7},   {6, 7},
        {8, 7},   {10, 7},  {12, 7},  {14, 7}, // Pixels  16 -  23
        {16, 7},  {18, 7},  {20, 7},  {22, 7},
        {24, 7},  {26, 7},  {28, 7},  {30, 7}, // Pixels  24 -  31
        {0, 8},   {2, 8},   {4, 8},   {6, 8},
        {8, 8},   {10, 8},  {12, 8},  {14, 8}, // Pixels  32 -  39
        {16, 8},  {18, 8},  {20, 8},  {22, 8},
        {24, 8},  {26, 8},  {28, 8},  {30, 8}, // Pixels  40 -  47
        {0, 9},   {2, 9},   {4, 9},   {6, 9},
        {8, 9},   {10, 9},  {12, 9},  {14, 9}, // Pixels  48 -  55
        {16, 9},  {18, 9},  {20, 9},  {22, 9},
        {24, 9},  {26, 9},  {28, 9},  {30, 9}, // Pixels  56 -  63
        {0, 10},  {2, 10},  {4, 10},  {6, 10},
        {8, 10},  {10, 10}, {12, 10}, {14, 10}, // Pixels  64 -  71
        {16, 10}, {18, 10}, {20, 10}, {22, 10},
        {24, 10}, {26, 10}, {28, 10}, {30, 10}, // Pixels  72 -  79
        {0, 11},  {2, 11},  {4, 11},  {6, 11},
        {8, 11},  {10, 11}, {12, 11}, {14, 11}, // Pixels  80 -  87
        {16, 11}, {18, 11}, {20, 11}, {22, 11},
        {24, 11}, {26, 11}, {28, 11}, {30, 11}, // Pixels  88 -  95
        {0, 12},  {2, 12},  {4, 12},  {6, 12},
        {8, 12},  {10, 12}, {12, 12}, {14, 12}, // Pixels  96 - 103
        {16, 12}, {18, 12}, {20, 12}, {22, 12},
        {24, 12}, {26, 12}, {28, 12}, {30, 12}, // Pixels 104 - 111
        {0, 13},  {2, 13},  {4, 13},  {6, 13},
        {8, 13},  {10, 13}, {12, 13}, {14, 13}, // Pixels 112 - 119
        {16, 13}, {18, 13}, {20, 13}, {22, 13},
        {24, 13}, {26, 13}, {28, 13}, {30, 13}, // Pixels 120 - 127
        {0, 14},  {2, 14},  {4, 14},  {6, 14},
        {8, 14},  {10, 14}, {12, 14}, {14, 14}, // Pixels 128 - 135
        {16, 14}, {18, 14}, {20, 14}, {22, 14},
        {24, 14}, {26, 14}, {28, 14}, {30, 14}, // Pixels 136 - 143
        {0, 15},  {2, 15},  {4, 15},  {6, 15},
        {8, 15},  {10, 15}, {12, 15}, {14, 15}, // Pixels 144 - 151
        {16, 15}, {18, 15}, {20, 15}, {22, 15},
        {24, 15}, {26, 15}, {28, 15}, {30, 15}, // Pixels 152 - 159
        {0, 16},  {2, 16},  {4, 16},  {6, 16},
        {8, 16},  {10, 16}, {12, 16}, {14, 16}, // Pixels 160 - 167
        {16, 16}, {18, 16}, {20, 16}, {22, 16},
        {24, 16}, {26, 16}, {28, 16}, {30, 16}, // Pixels 168 - 175
        {0, 17},  {2, 17},  {4, 17},  {6, 17},
        {8, 17},  {10, 17}, {12, 17}, {14, 17}, // Pixels 176 - 183
        {16, 17}, {18, 17}, {20, 17}, {22, 17},
        {24, 17}, {26, 17}, {28, 17}, {30, 17}, // Pixels 184 - 191
        {0, 18},  {2, 18},  {4, 18},  {6, 18},
        {8, 18},  {10, 18}, {12, 18}, {14, 18}, // Pixels 192 - 199
        {16, 18}, {18, 18}, {20, 18}, {22, 18},
        {24, 18}, {26, 18}, {28, 18}, {30, 18}, // Pixels 200 - 207
        {0, 19},  {2, 19},  {4, 19},  {6, 19},
        {8, 19},  {10, 19}, {12, 19}, {14, 19}, // Pixels 208 - 215
        {16, 19}, {18, 19}, {20, 19}, {22, 19},
        {24, 19}, {26, 19}, {28, 19}, {30, 19}, // Pixels 216 - 223
        {0, 20},  {2, 20},  {4, 20},  {6, 20},
        {8, 20},  {10, 20}, {12, 20}, {14, 20}, // Pixels 224 - 231
        {16, 20}, {18, 20}, {20, 20}, {22, 20},
        {24, 20}, {26, 20}, {28, 20}, {30, 20}, // Pixels 232 - 239
        {0, 21},  {2, 21},  {4, 21},  {6, 21},
        {8, 21},  {10, 21}, {12, 21}, {14, 21}, // Pixels 240 - 247
        {16, 21}, {18, 21}, {20, 21}, {22, 21},
        {24, 21}, {26, 21}, {28, 21}, {30, 21}, // Pixels 248 - 255
        {0, 22},  {2, 22},  {4, 22},  {6, 22},
        {8, 22},  {10, 22}, {12, 22}, {14, 22}, // Pixels 256 - 263
        {16, 22}, {18, 22}, {20, 22}, {22, 22},
        {24, 22}, {26, 22}, {28, 22}, {30, 22}, // Pixels 264 - 271
        {0, 23},  {2, 23},  {4, 23},  {6, 23},
        {8, 23},  {10, 23}, {12, 23}, {14, 23}, // Pixels 272 - 279
        {16, 23}, {18, 23}, {20, 23}, {22, 23},
        {24, 23}, {26, 23}, {28, 23}, {30, 23}, // Pixels 280 - 287
        {1, 0},   {3, 0},   {5, 0},   {7, 0},
        {9, 0},   {11, 0},  {13, 0},  {15, 0}, // Pixels 288 - 295
        {17, 0},  {19, 0},  {21, 0},  {23, 0},
        {25, 0},  {27, 0},  {29, 0},  {31, 0}, // Pixels 296 - 303
        {1, 1},   {3, 1},   {5, 1},   {7, 1},
        {9, 1},   {11, 1},  {13, 1},  {15, 1}, // Pixels 304 - 311
        {17, 1},  {19, 1},  {21, 1},  {23, 1},
        {25, 1},  {27, 1},  {29, 1},  {31, 1}, // Pixels 312 - 319
        {1, 2},   {3, 2},   {5, 2},   {7, 2},
        {9, 2},   {11, 2},  {13, 2},  {15, 2}, // Pixels 320 - 327
        {17, 2},  {19, 2},  {21, 2},  {23, 2},
        {25, 2},  {27, 2},  {29, 2},  {31, 2}, // Pixels 328 - 335
        {1, 3},   {3, 3},   {5, 3},   {7, 3},
        {9, 3},   {11, 3},  {13, 3},  {15, 3}, // Pixels 336 - 343
        {17, 3},  {19, 3},  {21, 3},  {23, 3},
        {25, 3},  {27, 3},  {29, 3},  {31, 3}, // Pixels 344 - 351
        {1, 4},   {3, 4},   {5, 4},   {7, 4},
        {9, 4},   {11, 4},  {13, 4},  {15, 4}, // Pixels 352 - 359
        {17, 4},  {19, 4},  {21, 4},  {23, 4},
        {25, 4},  {27, 4},  {29, 4},  {31, 4}, // Pixels 360 - 367
        {1, 5},   {3, 5},   {5, 5},   {7, 5},
        {9, 5},   {11, 5},  {13, 5},  {15, 5}, // Pixels 368 - 375
        {17, 5},  {19, 5},  {21, 5},  {23, 5},
        {25, 5},  {27, 5},  {29, 5},  {31, 5}, // Pixels 376 - 383
        {1, 6},   {3, 6},   {5, 6},   {7, 6},
        {9, 6},   {11, 6},  {13, 6},  {15, 6}, // Pixels 384 - 391
        {17, 6},  {19, 6},  {21, 6},  {23, 6},
        {25, 6},  {27, 6},  {29, 6},  {31, 6}, // Pixels 392 - 399
        {1, 7},   {3, 7},   {5, 7},   {7, 7},
        {9, 7},   {11, 7},  {13, 7},  {15, 7}, // Pixels 400 - 407
        {17, 7},  {19, 7},  {21, 7},  {23, 7},
        {25, 7},  {27, 7},  {29, 7},  {31, 7}, // Pixels 408 - 415
        {1, 8},   {3, 8},   {5, 8},   {7, 8},
        {9, 8},   {11, 8},  {13, 8},  {15, 8}, // Pixels 416 - 423
        {17, 8},  {19, 8},  {21, 8},  {23, 8},
        {25, 8},  {27, 8},  {29, 8},  {31, 8}, // Pixels 424 - 431
        {1, 9},   {3, 9},   {5, 9},   {7, 9},
        {9, 9},   {11, 9},  {13, 9},  {15, 9}, // Pixels 432 - 439
        {17, 9},  {19, 9},  {21, 9},  {23, 9},
        {25, 9},  {27, 9},  {29, 9},  {31, 9}, // Pixels 440 - 447
        {1, 10},  {3, 10},  {5, 10},  {7, 10},
        {9, 10},  {11, 10}, {13, 10}, {15, 10}, // Pixels 448 - 455
        {17, 10}, {19, 10}, {21, 10}, {23, 10},
        {25, 10}, {27, 10}, {29, 10}, {31, 10}, // Pixels 456 - 463
        {1, 11},  {3, 11},  {5, 11},  {7, 11},
        {9, 11},  {11, 11}, {13, 11}, {15, 11}, // Pixels 464 - 471
        {17, 11}, {19, 11}, {21, 11}, {23, 11},
        {25, 11}, {27, 11}, {29, 11}, {31, 11}, // Pixels 472 - 479
        {1, 12},  {3, 12},  {5, 12},  {7, 12},
        {9, 12},  {11, 12}, {13, 12}, {15, 12}, // Pixels 480 - 487
        {17, 12}, {19, 12}, {21, 12}, {23, 12},
        {25, 12}, {27, 12}, {29, 12}, {31, 12}, // Pixels 488 - 495
        {1, 13},  {3, 13},  {5, 13},  {7, 13},
        {9, 13},  {11, 13}, {13, 13}, {15, 13}, // Pixels 496 - 503
        {17, 13}, {19, 13}, {21, 13}, {23, 13},
        {25, 13}, {27, 13}, {29, 13}, {31, 13}, // Pixels 504 - 511
        {1, 14},  {3, 14},  {5, 14},  {7, 14},
        {9, 14},  {11, 14}, {13, 14}, {15, 14}, // Pixels 512 - 519
        {17, 14}, {19, 14}, {21, 14}, {23, 14},
        {25, 14}, {27, 14}, {29, 14}, {31, 14}, // Pixels 520 - 527
        {1, 15},  {3, 15},  {5, 15},  {7, 15},
        {9, 15},  {11, 15}, {13, 15}, {15, 15}, // Pixels 528 - 535
        {17, 15}, {19, 15}, {21, 15}, {23, 15},
        {25, 15}, {27, 15}, {29, 15}, {31, 15}, // Pixels 536 - 543
        {1, 16},  {3, 16},  {5, 16},  {7, 16},
        {9, 16},  {11, 16}, {13, 16}, {15, 16}, // Pixels 544 - 551
        {17, 16}, {19, 16}, {21, 16}, {23, 16},
        {25, 16}, {27, 16}, {29, 16}, {31, 16}, // Pixels 552 - 559
        {1, 17},  {3, 17},  {5, 17},  {7, 17},
        {9, 17},  {11, 17}, {13, 17}, {15, 17}, // Pixels 560 - 567
        {17, 17}, {19, 17}, {21, 17}, {23, 17},
        {25, 17}, {27, 17}, {29, 17}, {31, 17}, // Pixels 568 - 575
        {1, 18},  {3, 18},  {5, 18},  {7, 18},
        {9, 18},  {11, 18}, {13, 18}, {15, 18}, // Pixels 576 - 583
        {17, 18}, {19, 18}, {21, 18}, {23, 18},
        {25, 18}, {27, 18}, {29, 18}, {31, 18}, // Pixels 584 - 591
        {1, 19},  {3, 19},  {5, 19},  {7, 19},
        {9, 19},  {11, 19}, {13, 19}, {15, 19}, // Pixels 592 - 599
        {17, 19}, {19, 19}, {21, 19}, {23, 19},
        {25, 19}, {27, 19}, {29, 19}, {31, 19}, // Pixels 600 - 607
        {1, 20},  {3, 20},  {5, 20},  {7, 20},
        {9, 20},  {11, 20}, {13, 20}, {15, 20}, // Pixels 608 - 615
        {17, 20}, {19, 20}, {21, 20}, {23, 20},
        {25, 20}, {27, 20}, {29, 20}, {31, 20}, // Pixels 616 - 623
        {1, 21},  {3, 21},  {5, 21},  {7, 21},
        {9, 21},  {11, 21}, {13, 21}, {15, 21}, // Pixels 624 - 631
        {17, 21}, {19, 21}, {21, 21}, {23, 21},
        {25, 21}, {27, 21}, {29, 21}, {31, 21}, // Pixels 632 - 639
        {1, 22},  {3, 22},  {5, 22},  {7, 22},
        {9, 22},  {11, 22}, {13, 22}, {15, 22}, // Pixels 640 - 647
        {17, 22}, {19, 22}, {21, 22}, {23, 22},
        {25, 22}, {27, 22}, {29, 22}, {31, 22}, // Pixels 648 - 655
        {1, 23},  {3, 23},  {5, 23},  {7, 23},
        {9, 23},  {11, 23}, {13, 23}, {15, 23}, // Pixels 656 - 663
        {17, 23}, {19, 23}, {21, 23}, {23, 23},
        {25, 23}, {27, 23}, {29, 23}, {31, 23} // Pixels 664 - 671
};

/******************************************************************************
 *  format_printhead_data_KGE()
 *      Processes data line by line and places it into the Magicard
 *Alto/Opera/Tempo
 *      or Enduro Printhead Format.
 *
 *  Parameters:
 *      psettings     Pointer to the settings struct.
 *      lpData        Input Data Buffer
 *      lpOutBuff     Output Data Buffer
 *      nNoOfBits     Colour or Monochrome data
 *      nSize         Size of Output Data Buffer
 *
 *  Returns:
 *     None
 *
 ******************************************************************************/

void format_printhead_data_KGE(PSETTINGS psettings,
                               uint8_t * lpData,
                               uint8_t * lpOutBuff,
                               int       nNoOfBits,
                               int       nSize) {
    int       margin_index;
    int       line_length = 0;
    int       pixel       = 0;
    int       first_word;
    int       first_bit;
    int       change_point = PIXELS_PART_1;
    int       print_bit;
    int       which_bit = 0;
    int       a;
    uint32_t  temp_word[BITS_FOR_COLOUR_DATA];
    uint32_t  density;
    uint32_t  dwtemp;
    uint32_t *temp = NULL;

    int nPrintheadPosition =
            GetPrintheadPosition(psettings) - psettings->nImagePosition_UpDown;
    memset(lpOutBuff, 0, nSize);

    if (psettings->pageType == 3 /*DMPAPER_WHITEBORDER*/) {
        nPrintheadPosition = MIN(nPrintheadPosition, BORDER_CARD_MAX_MARGIN);
        nPrintheadPosition = MAX(nPrintheadPosition, BORDER_CARD_MIN_MARGIN);
        line_length        = BORDER_CARD_WIDTH;
        margin_index       = BORDER_CARD_MAX_MARGIN - nPrintheadPosition;
    } else {
        nPrintheadPosition = MIN(nPrintheadPosition, FULL_BLEED_MAX_MARGIN);
        nPrintheadPosition = MAX(nPrintheadPosition, FULL_BLEED_MIN_MARGIN);
        line_length        = FULL_BLEED_WIDTH;
        margin_index       = FULL_BLEED_MAX_MARGIN - nPrintheadPosition;
    }

    if (nNoOfBits > 1) {
        while (pixel < line_length) {
            first_word = print_format[margin_index + pixel].first_word;
            first_bit  = print_format[margin_index + pixel].first_bit;
            temp       = (uint32_t *)(lpOutBuff + (first_word * 4));

            while ((pixel < line_length) &&
                   ((margin_index + pixel) < change_point)) {
                memset((uint8_t *)temp_word, 0, sizeof(temp_word));

                for (print_bit = first_bit;
                     ((margin_index + pixel) < change_point) &&
                     (print_bit < 32) && (pixel < line_length);
                     print_bit += 2) {
                    pixel++;
                    density = *lpData++;
                    temp_word[0] |= ((density & 0x01) << print_bit);
                    density >>= 1;
                    temp_word[1] |= ((density & 0x01) << print_bit);
                    density >>= 1;
                    temp_word[2] |= ((density & 0x01) << print_bit);
                    density >>= 1;
                    temp_word[3] |= ((density & 0x01) << print_bit);
                    density >>= 1;
                    temp_word[4] |= ((density & 0x01) << print_bit);
                    density >>= 1;
                    temp_word[5] |= ((density & 0x01) << print_bit);
                }

                *temp |= temp_word[0];
                temp += 24;
                *temp |= temp_word[1];
                temp += 24;
                *temp |= temp_word[2];
                temp += 24;
                *temp |= temp_word[3];
                temp += 24;
                *temp |= temp_word[4];
                temp += 24;
                *temp |= temp_word[5];
                temp -= 119; // - ((24 * 5) - 1)

                first_bit = which_bit;

            } // end while ((margin_index + pixel ) < change_point)

            change_point = margin_index + line_length;
            which_bit    = 1;

        } // end while  y_index < line_length
    } else {

        uint8_t bSrcBit = 0x80;

        // 1 bit resin data
        while (pixel < line_length) {
            first_word = print_format[margin_index + pixel].first_word;
            first_bit  = print_format[margin_index + pixel].first_bit;
            temp       = (uint32_t *)(lpOutBuff + (first_word * 4));

            while ((pixel < line_length) &&
                   ((margin_index + pixel) < change_point)) {
                temp_word[0] = 0;
                for (print_bit = first_bit;
                     ((margin_index + pixel) < change_point) &&
                     (print_bit < 32) && (pixel < line_length);
                     print_bit += 2) {

                    if (*lpData & bSrcBit) {
                        temp_word[0] |= 0x01 << print_bit;
                    }

                    if (bSrcBit == 0x01) {
                        lpData++;
                        bSrcBit = 0x80;
                    } else {
                        bSrcBit >>= 1;
                    }

                    pixel++;
                }
                *temp |= temp_word[0];
                first_bit = which_bit;
                temp++;
            }
            change_point = margin_index + line_length;
            which_bit    = 1;
        } // end for printhead_section
    }

    if (ENDURO_OEM(psettings)) {
        for (a = 0; a < (nNoOfBits * 24); a++) {
            temp  = (uint32_t *)(lpOutBuff + (a * 4));
            *temp = (((*temp & 0xaaaaaaaa) >> 1) | ((*temp & 0x55555555) << 1));
            *temp = (((*temp & 0xcccccccc) >> 2) | ((*temp & 0x33333333) << 2));
            *temp = (((*temp & 0xf0f0f0f0) >> 4) | ((*temp & 0x0f0f0f0f) << 4));
        }
    } else {
        // Now place the 4 bytes in the right order
        for (a = 0; a < (nNoOfBits * 24); a++) {
            dwtemp = 0;
            temp   = (uint32_t *)(lpOutBuff + (a * 4));
            dwtemp |= *temp >> 24;
            dwtemp |= (*temp >> 8) & 0x0000FF00;
            dwtemp |= (*temp << 8) & 0x00FF0000;
            dwtemp |= (*temp << 24) & 0xFF000000;
            *temp = dwtemp;
        }
    }
}

/******************************************************************************
 *  format_printhead_data()
 *      Determines which printhead format is required and calls the appropriate
 *      function.
 *
 *  Parameters:
 *      psettings    Pointer to the settings
 *      lpData       Input Data Buffer
 *      lpOutBuff    Output Data Buffer
 *      nNoOfBits    Colour or Monochrome data
 *      nSize        Size of dOutput Data Buffer
 *
 *  Returns:
 *     None
 ******************************************************************************/

void format_printhead_data(PSETTINGS psettings,
                           uint8_t * lpData,
                           uint8_t * lpOutBuff,
                           int       nNoOfBits,
                           int       nSize) {

    if (RIO_OEM(psettings) || AVALON_MODEL(psettings)) {
        format_printhead_data_KEE(
                psettings, lpData, lpOutBuff, nNoOfBits, nSize);
    } else {
        format_printhead_data_KGE(
                psettings, lpData, lpOutBuff, nNoOfBits, nSize);
    }
}

/******************************************************************************
 *
 *  SharpenImage()
 *
 *  Apply the following convolution filter to the input image:
 *
 *        /  0.25 -1.00  0.25 \     / 0.50 \
 *        | -1.00  4.00 -1.00 |  =  | 2.00 | * (0.50  2.00  0.50)
 *        \  0.25 -1.00  0.25 /     \ 0.50 /
 *
 *  Since the 2D filter is xy-separable, we apply two 1D-filters in the
 *  vertical and horizontal directions.  This is computationally more
 *  efficient.
 *
 *  Parameters:
 *      pdev            Pointer to our PDEV
 *      ulSharpLevel    The level of sharpening desired.
 *      b6bit           clamp to 64 level output or not
 *
 *  Returns:
 *    None.
 *
 *  Note: I have been a little lazy.  The corner pixels at (0,0), (0,height),
 *  (width,0) and (width,height) don't get altered by this algorithm.  This has
 *  the benefit of making the code much easier to follow.  If we put this into
 *  production code then this issue may have to be addressed.
 *
 *****************************************************************************/
typedef enum {
    SHARPNESS_NONE = -2,
    SHARPNESS_LESS,
    SHARPNESS_NORMAL,
    SHARPNESS_MORE,
    SHARPNESS_MAX
} SHARPNESS;

void SharpenImage(PDEVDATA pdev, SHARPNESS ulSharpLevel, bool b6bit) {
    int32_t  lWidth       = pdev->ulCMYInPlaneWidth;
    int32_t  lHeight      = pdev->yImage;
    uint8_t *lpCurrent    = NULL;
    uint8_t *lpNext       = NULL;
    uint32_t ulColour     = 0;
    uint8_t  PrevInPixVal = 0;
    int32_t  lX           = 0;
    int32_t  lY           = 0;

    // larger than uint8_t so we can detect overflow
    int32_t lOutPixVal       = 0;
    int32_t lWeightCurrent   = 1;
    int32_t lWeightNeighbour = 0;
    int32_t lWeightDivisor   = 1;

    // If we have no YMC data (i.e K Plane only) then we do no sharpening
    if (lWidth == 0) {
        ////ERROR_MSG(pdev->lpdm, TEXT("### Sharpening - No data to sharpen
        ///###\n"));
        return;
    }

    // set up sharpening parameters
    switch (ulSharpLevel) {
        case SHARPNESS_LESS:
            // ERROR_MSG(pdev->lpdm, TEXT("### Sharpening - Less ###\n"));

            // subtle sharpnening (image may be slightly soft-looking, but not
            // as much as an unprocessed image)
            lWeightCurrent   = 6;
            lWeightNeighbour = 1;
            lWeightDivisor   = 4;
            break;

        case SHARPNESS_NORMAL:
            // ERROR_MSG(pdev->lpdm, TEXT("### Sharpening - Normal ###\n"));

            // 'normal' sharpening (best level for images which contain lots of
            // fine detail)
            lWeightCurrent   = 4;
            lWeightNeighbour = 1;
            lWeightDivisor   = 2;
            break;

        case SHARPNESS_MORE:
            // ERROR_MSG(pdev->lpdm, TEXT("### Sharpening - More ###\n"));

            // strong sharpening (if the source image is slightly blurred and
            // needs 'fixing', then this may help
            lWeightCurrent   = 10;
            lWeightNeighbour = 3;
            lWeightDivisor   = 4;
            break;

        case SHARPNESS_MAX:
            // ERROR_MSG(pdev->lpdm, TEXT("### Sharpening - Max ###\n"));

            // maximum sharpening (this might 'fix' blurry photos, but will look
            // dreadful on most images)
            lWeightCurrent   = 3;
            lWeightNeighbour = 1;
            lWeightDivisor   = 1;
            break;

        default:
            // ERROR_MSG(pdev->lpdm, TEXT("### Sharpening - None ###\n"));

            // no sharpening
            return;
    }

    for (ulColour = 0; ulColour < 3; ++ulColour) {
        // convolve in vertical direction (y) for each x-value
        for (lX = 0; lX < lWidth; ++lX) {
            lpCurrent = pdev->lpPageBuffer[ulColour] + lX + lWidth;
            lpNext    = lpCurrent + lWidth;

            PrevInPixVal = *(lpCurrent - lWidth);

            for (lY = 1; lY < lHeight - 1; ++lY) {
                lOutPixVal = ((lWeightCurrent * (int32_t)*lpCurrent) -
                              (lWeightNeighbour *
                               ((int32_t)*lpNext + (int32_t)PrevInPixVal))) /
                             lWeightDivisor;
                PrevInPixVal = *lpCurrent;

                if (b6bit == false) {
                    // clamp to [0,255] if necessary
                    *lpCurrent = (lOutPixVal > 255) ? (uint8_t)255 :
                                         (lOutPixVal < 0) ? (uint8_t)0 :
                                                            (uint8_t)lOutPixVal;
                } else {
                    *lpCurrent = (lOutPixVal > 63) ? (uint8_t)63 :
                                 (lOutPixVal < 0)  ? (uint8_t)0 :
                                                (uint8_t)lOutPixVal;
                }
                // advance to next set of pixels
                lpCurrent += lWidth;
                lpNext += lWidth;
            }
        }

        // convolve in the horizontal direction (x) for each y-value
        for (lY = 0; lY < lHeight; ++lY) {
            lpCurrent = pdev->lpPageBuffer[ulColour] + (lWidth * lY) + 1;
            lpNext    = lpCurrent + 1;

            PrevInPixVal = *(lpCurrent - 1);

            for (lX = 1; lX < lWidth - 1; ++lX) {
                lOutPixVal = ((lWeightCurrent * (int32_t)*lpCurrent) -
                              (lWeightNeighbour *
                               ((int32_t)*lpNext + (int32_t)PrevInPixVal))) /
                             lWeightDivisor;
                PrevInPixVal = *lpCurrent;

                if (b6bit == false) {
                    // clamp to [0,255] if necessary
                    *lpCurrent = (lOutPixVal > 255) ? (uint8_t)255 :
                                         (lOutPixVal < 0) ? (uint8_t)0 :
                                                            (uint8_t)lOutPixVal;
                } else {
                    *lpCurrent = (lOutPixVal > 63) ? (uint8_t)63 :
                                 (lOutPixVal < 0)  ? (uint8_t)0 :
                                                (uint8_t)lOutPixVal;
                }
                // advance to next set of pixels
                ++lpCurrent;
                ++lpNext;
            }
        }
    }
}

/*******************************************************************************
 *  CopyColourPlaneToBuffer()
 *      Receive plane and write it on the buffer.
 *
 *  Parameters:
 *      pdev         Pointer to our PDEVICE
 *      psettings
 *      plSrc        Pointer to the image data. This can be packed or planar.
 *      ulColour     Colour index
 *      bPlanar      The pixel order in lpSrc. true=Planar, false=Packed
 *
 *  Returns:
 *      None
 *******************************************************************************/
void CopyColourPlaneToBuffer(PDEVDATA  pdev,
                             PSETTINGS psettings,
                             uint8_t * lpSrc,
                             uint32_t  ulColour,
                             bool      bPlanar) {
    uint8_t *lpPlane      = NULL;
    bool     bDataInPlane = false;
    uint32_t x            = 0;
    uint32_t ulXMove      = 0;
    uint32_t ulDataWidth  = 0;

    lpPlane = pdev->lpCMYIn[ulColour];

    if (RIO_OEM(psettings) || AVALON_MODEL(psettings)) {
        ulDataWidth = OUTPUT_DATA_WIDTH_RIO;
    } else if (PRO360_OEM(psettings)) {
        // OUTPUT_DATA_WIDTH_PRO360;
        ulDataWidth = pdev->xImage;
    } else if (HELIX_OEM(psettings)) {
        ulDataWidth = OUTPUT_DATA_WIDTH_HELIX;
    } else {
        ulDataWidth = OUTPUT_DATA_WIDTH_ALTO;
    }

    if (bPlanar) {
        ulXMove = 1;
    } else {
        ulXMove = 4;
    }

    for (x = 0; x < pdev->ulCMYInPlaneWidth; x++) {
        *lpPlane++ = *lpSrc;
        if (*lpSrc != 0x00) {
            bDataInPlane = true;
        }

        lpSrc += ulXMove;
    }

    if (bDataInPlane) {
        // Convert source strip into device format
        if (PRO360_OEM(psettings)) {
            CopyToBuffer(
                    pdev, pdev->lpCMYIn[ulColour], ulDataWidth, ulColour, true);
        } else if (HELIX_OEM(psettings)) {
            CopyToBuffer(
                    pdev, pdev->lpCMYIn[ulColour], ulDataWidth, ulColour, true);
        } else {
            format_printhead_data(psettings,
                                  pdev->lpCMYIn[ulColour],
                                  pdev->lpCMYOut[ulColour],
                                  BITS_FOR_COLOUR_DATA,
                                  pdev->ulCMYOutPlaneWidth);
            CopyToBuffer(pdev,
                         pdev->lpCMYOut[ulColour],
                         ulDataWidth,
                         ulColour,
                         true);
        }
    } else {
        /*******************************************************************************
         * If scanline does not have any colour, copy white pixels.
         * Because device does not have any command to move position, we still
         *need to keep white datas.
         *******************************************************************************/
        CopyToBuffer(
                pdev, pdev->lpCMYIn[ulColour], ulDataWidth, ulColour, false);
    }

    return;
}

void CopyYMCPageBuffToOutputBuff(PDEVDATA pdev, PSETTINGS psettings) {
    // Width of the CMY plane (may include padding)
    uint32_t lWidth    = pdev->ulCMYInPlaneWidth;
    int16_t  y         = 0;
    uint32_t ulColour  = 0;
    uint8_t *lpCurrent = NULL;

    // Copy colour plane to the output buffer.
    for (y = 0; y < pdev->yImage; y++) {
        for (ulColour = 0; ulColour < 3; ulColour++) {
            lpCurrent = pdev->lpPageBuffer[ulColour] + (lWidth * y);
            CopyColourPlaneToBuffer(pdev, psettings, lpCurrent, ulColour, true);
        }
    }
}

/*******************************************************************************
 *  DetectAdjacentColour()
 *      Copy SURFOBJ which contains K-resin object information recorded during
 *journaling.
 *      Also rotate it depends on the orientation setting.
 *
 *  Parameters:
 *      pdev         Pointer to our PDEV.
 *      pSettings    Pointer to our settings.
 *
 *  Returns:
 *      None
 *******************************************************************************/
bool DetectAdjacentColour(PDEVDATA pdev, PSETTINGS pSettings) {
    uint8_t *lpKPlane  = NULL;
    uint8_t *lpPrev    = NULL;
    uint8_t *lpCurrent = NULL;
    uint8_t *lpNext    = NULL;

    // Width of the CMY plane (may include padding)
    int32_t  lWidth   = pdev->ulCMYInPlaneWidth;
    int32_t  lHeight  = pdev->yImage;
    int32_t  x        = 0;
    int32_t  y        = 0;
    int32_t  lXMove   = 0;
    uint32_t ulColour = 0;

    if (pSettings->OEM != OEM_HELIX && pSettings->OEM != OEM_PRO360) {
        DetectEdge(pdev);
    }

    /*******************************************************************************
     * Start from the bottom of the bitmap and work upwards since this is more
     * suitable to counteract the physical registration problems of the device
     *
     * lpCurrent is used to point to the pixel that we wish to process
     * lpPrev points to the pixel directly above lpCurrent
     * lpNext points to the pixel directly below lpCurrent
     *******************************************************************************/

    // --- The Bottom line ---  NB 'Bottom line' is the last line of the image
    for (ulColour = 0; ulColour < 3; ulColour++) {
        lpKPlane  = pdev->lpPageBuffer[PLANE_HALO] + lWidth * (lHeight - 1);
        lpCurrent = pdev->lpPageBuffer[ulColour] + lWidth * (lHeight - 1);
        lpPrev    = lpCurrent - lWidth;

        for (x = 0; x < lWidth; x++) {
            if (*lpKPlane != 0x0) {
                if (x == 0) {
                    // The left most pixel in the strip.
                    ReplaceYMCPixelColour(lpPrev, lpCurrent, NULL);
                } else if (x + 1 == lWidth) {
                    // The right most pixel in the strip.
                    ReplaceYMCPixelColour(lpPrev, lpCurrent, NULL);
                } else {
                    // Somewhere in the middle of the strip
                    ReplaceYMCPixelColour(lpPrev, lpCurrent, NULL);
                }
            }

            // Move along the scanline
            lpKPlane++;
            lpPrev++;
            lpCurrent++;
        }
    }

    // ---- Middle lines ----
    for (ulColour = 0; ulColour < 3; ulColour++) {
        lXMove = 1;

        // From (height - 1) to the second line
        for (y = lHeight - 1; y != 1; y--) {
            lpKPlane  = pdev->lpPageBuffer[PLANE_HALO] + (lWidth * y);
            lpCurrent = pdev->lpPageBuffer[ulColour] + (lWidth * y);
            lpPrev    = lpCurrent - lWidth;
            lpNext    = lpCurrent + lWidth;

            if (lXMove == -1) {
                // Do next search from left to right
                lXMove = 1;
            } else {
                // Do next search from right to left
                lXMove = -1;

                // Move pointers to the right edge.
                lpKPlane += lWidth - 1;
                lpCurrent += lWidth - 1;
                lpPrev += lWidth - 1;
                lpNext += lWidth - 1;
            }

            for (x = 0; x < lWidth; x++) {
                if (*lpKPlane != 0x0) {
                    if (x == 0) {
                        // The left pixel in the strip.
                        ReplaceYMCPixelColour(lpPrev, lpCurrent, lpNext);
                    } else if (x + 1 == lWidth) {
                        // The right pixel in the strip.
                        ReplaceYMCPixelColour(lpPrev, lpCurrent, lpNext);
                    } else {
                        // Somewhere in the middle of the strip
                        ReplaceYMCPixelColour(lpPrev, lpCurrent, lpNext);
                    }
                }

                // Move along the scanline in the desired direction
                lpKPlane += lXMove;
                lpCurrent += lXMove;
                lpPrev += lXMove;
                lpNext += lXMove;
            }
        }
    }

    // --- The top line ---  NB 'Top Line' is the first line of the image
    for (ulColour = 0; ulColour < 3; ulColour++) {
        lpKPlane  = pdev->lpPageBuffer[PLANE_HALO];
        lpCurrent = pdev->lpPageBuffer[ulColour];
        lpNext    = lpCurrent + lWidth;

        for (x = 0; x < lWidth; x++) {
            // lpNext points the pixel under the K-resin object.
            if (*lpKPlane != 0x0) {
                if (x == 0) {
                    // The left pixel in the strip.
                    ReplaceYMCPixelColour(NULL, lpCurrent, lpNext);
                } else if (x + 1 == lWidth) {
                    // The right pixel in the strip.
                    ReplaceYMCPixelColour(NULL, lpCurrent, lpNext);
                } else {
                    // Somewhere in the middle of the strip
                    ReplaceYMCPixelColour(NULL, lpCurrent, lpNext);
                }
            }

            // Move along the scanline
            lpKPlane++;
            lpCurrent++;
            lpNext++;
        }
    }

    SharpenImage(pdev,
                 pSettings->nSharpness,
                 (pSettings->OEM == OEM_HELIX || pSettings->OEM == OEM_PRO360) ?
                         false :
                         true);

    // Copy colour plane to the output buffer.
    for (y = 0; y < pdev->yImage; y++) {

        CopyKPlaneToBuffer(pdev,
                           pdev->lpPageBuffer[PLANE_K] + (lWidth * y),
                           pdev->lpKSrcStrip,
                           pSettings,
                           y);

        for (ulColour = 0; ulColour < 3; ulColour++) {
            lpCurrent = pdev->lpPageBuffer[ulColour] + (lWidth * y);
            CopyColourPlaneToBuffer(pdev, pSettings, lpCurrent, ulColour, true);
        }
    }
    TRACE_OUT;
    return true;
}

void endPage(PDEVDATA            pdev,
             PSETTINGS           psettings,
             cups_page_header2_t header,
             bool sharpen) {

    TRACE_IN;
    TRACE("pdev->eChannelOption %u", pdev->eChannelOption);

    if (sharpen) {
       // Check whether the current page requires the "Detect adjacent colour"
       // option to be processed
        if ((pdev->eChannelOption == UICBVAL_YMCK) &&
            pdev->bDetectAdjacentColour) {
            // Modify colours underneath the edges of black areas to mask any
            // colour mis-registration Sharpening will then occur afterwards
          pdev->iHaloRadius = 25;
          DetectAdjacentColour(pdev, psettings);
       } else if (pdev->eChannelOption == UICBVAL_YMC) {
            SharpenImage(pdev,
                psettings->nSharpness,
                         (psettings->OEM == OEM_HELIX ||
                          psettings->OEM == OEM_PRO360) ?
                false :
                true);
          CopyYMCPageBuffToOutputBuff(pdev, psettings);
       }
    }

    TRACE("pdev->bDuplex = %u, pdev->bFrontPage = %u",
           pdev->bDuplex,
           pdev->bFrontPage);

    if (pdev->bDuplex && pdev->bFrontPage) {
        // Skip output page data to wait for next page
        TRACE_STR("Skip output page data to wait for next page 1");

    } else if (pdev->bDuplex && !pdev->bFrontPage) {

        // Output Front Page
        TRACE_STR("Output Front Page 1");

        if (psettings->OEM == OEM_PRO360) {
            OutputPro360HeaderCommand(pdev, header, psettings, true);
        } else if (psettings->OEM == OEM_HELIX) {
            OutputHelixHeaderCommand(pdev, header, psettings, true);
        } else {
            OutputHeaderCommand(pdev, header, psettings, true);
        }

        if (!IgnoreImageData(psettings)) {
            if (psettings->OEM != OEM_HELIX && psettings->OEM != OEM_PRO360) {
                SetDataSize(pdev, true);
            }
            FlushBuffer(pdev, psettings, true);
        }

        FreeBuffer(pdev, true);

        // Output Back Page
        TRACE_STR("Output Back Page : 1");
        if (psettings->OEM != OEM_HELIX && psettings->OEM != OEM_PRO360) {
            OutputHeaderCommand(pdev, header, psettings, false);
        }

        if (!IgnoreImageData(psettings)) {
            if (psettings->OEM != OEM_HELIX && psettings->OEM != OEM_PRO360) {
                SetDataSize(pdev, false);
            }

            TRACE_STR("Output Back Page : 1");
            FlushBuffer(pdev, psettings, false);
            TRACE_STR("returned from flushbuffer : 1");
        }

        if (psettings->OEM == OEM_PRO360) {
            pdev->epCheckSum = pdev->epCheckSum ^ 0xffffffffu;
            myWrite((char *)&pdev->epCheckSum, 4);
        }
        FreeBuffer(pdev, false);
    } else {
        // Simplex case
        TRACE_STR("Simplex case : 1");

        // Send out header command
        switch (psettings->OEM) {
            case OEM_PRO360: {
                TRACE_STR("Calling OutputPro360HeaderCommand");
                OutputPro360HeaderCommand(
                        pdev, header, psettings, pdev->bFrontPage);
                break;
            }
            case OEM_HELIX: {
                TRACE_STR("Calling OutputHelixHeaderCommand ");
                OutputHelixHeaderCommand(
                        pdev, header, psettings, pdev->bFrontPage);
                break;
            }
            default: {
                TRACE_STR("Calling OutputHeaderCommand\n ");
                OutputHeaderCommand(pdev, header, psettings, pdev->bFrontPage);
                break;
            }
        }

        // When magnetic encoding only option is set, we will not send any page
        // data.
        if (!IgnoreImageData(psettings)) {
            // Send out plane data size command
            if (psettings->OEM != OEM_HELIX && psettings->OEM != OEM_PRO360)
                SetDataSize(pdev, pdev->bFrontPage);

            TRACE_AT;
            // Send out plane data which we saved during dvOutputStrip
            FlushBuffer(pdev, psettings, pdev->bFrontPage);
            TRACE_AT;
        }
        if (psettings->OEM == OEM_PRO360) {
            pdev->epCheckSum = pdev->epCheckSum ^ 0xffffffffu;
            myWrite((char *)&pdev->epCheckSum, 4);
        }
        // Clean up memory for buffering
        TRACE_AT;
        FreeBuffer(pdev, pdev->bFrontPage);
        TRACE_AT;
    }

    // Clean buffers
    TRACE_AT;
    DeleteOutputBuffer(pdev);

    // Cleanup halftoning information
    if ((pdev->eChannelOption == UICBVAL_YMCK) ||
        (pdev->eChannelOption == UICBVAL_KResin)) {
        DeleteKPrintingObject(pdev);
    }

    fflush(stdout);
    TRACE_OUT;
}

void endJob(PDEVDATA            pdev,
            struct settings_    settings,
            cups_page_header2_t header) {
    // odd page duplex jobs we need to send the last page simplex
    if (pdev->bDuplex && settings.iTotalJobPages == 0 &&
        (pdev->iPageNumber % 2)) {
        pdev->bDuplex = false;
        // Output back page
        if (settings.OEM != OEM_HELIX && settings.OEM != OEM_PRO360) {
            OutputHeaderCommand(
                    pdev, header, &settings, true /*pdev->bFrontPage*/);
        }

        if (!IgnoreImageData(&settings)) {
            // Send out plane data size command
            if (settings.OEM != OEM_HELIX && settings.OEM != OEM_PRO360) {
                SetDataSize(pdev, true);
            }

            FlushBuffer(pdev, &settings, pdev->bFrontPage);
        }
        if (settings.OEM == OEM_PRO360) {
            pdev->epCheckSum = pdev->epCheckSum ^ 0xffffffffu;
            myWrite((char *)&pdev->epCheckSum, 4);
        }
        // Clean up memory for buffering
        FreeBuffer(pdev, pdev->bFrontPage);
    }
}

#define GET_LIB_FN_OR_EXIT_FAILURE(fn_ptr, lib, fn_name)                       \
    {                                                                          \
        fn_ptr = dlsym(lib, fn_name);                                          \
        if ((dlerror()) != NULL) {                                             \
            fputs("ERROR: required fn not exported from dynamically loaded "   \
                  "libary\n",                                                  \
                  stderr);                                                     \
            if (libCupsImage != 0)                                             \
                dlclose(libCupsImage);                                         \
            if (libCups != 0)                                                  \
                dlclose(libCups);                                              \
            return EXIT_FAILURE;                                               \
        }                                                                      \
    }

#ifdef RPMBUILD
#define CLEANUP                                                                \
    {                                                                          \
        if (originalRasterDataPtr != NULL) {                                   \
            free(originalRasterDataPtr);                                       \
        }                                                                      \
        CUPSRASTERCLOSE(ras);                                                  \
        if (fd != 0) {                                                         \
            close(fd);                                                         \
        }                                                                      \
        dlclose(libCupsImage);                                                 \
        dlclose(libCups);                                                      \
    }
#else
#define CLEANUP                                                                \
    {                                                                          \
        if (originalRasterDataPtr != NULL) {                                   \
            free(originalRasterDataPtr);                                       \
        }                                                                      \
        CUPSRASTERCLOSE(ras);                                                  \
        if (fd != 0) {                                                         \
            close(fd);                                                         \
        }                                                                      \
    }
#endif // defined RPMBUILD

uint32_t RGB_to_CMY(uint32_t *pColor) {
    uint8_t r, g, b, c, m, y;

    r = GetRValue(*pColor);
    g = GetGValue(*pColor);
    b = GetBValue(*pColor);

    c = 255 - r;
    m = 255 - g;
    y = 255 - b;

    return (CMYK(c, m, y, 0));
}

uint32_t RGB_to_CMYK(uint32_t *pColor) {
    uint8_t r, g, b, c, m, y;

    if (*pColor == 0) {
        return (CMYK(0, 0, 0, 255));
    }

    r = GetRValue(*pColor);
    g = GetGValue(*pColor);
    b = GetBValue(*pColor);

    c = 255 - r;
    m = 255 - g;
    y = 255 - b;

    return (CMYK(c, m, y, 0));
}

void error_diffusion(PDEVDATA pdev, uint8_t *pSurface) {
#define RAND(RN) (((seed = 1103515245 * seed + 12345) >> 12) % (RN))
#define INITERR(X, Y)                                                          \
    (((int)X) - (((int)Y) ? WHITE : BLACK) + ((WHITE / 2) - ((int)X)) / 2)
#define WHITE 255
#define BLACK 0

    uint8_t *pOutputImage = calloc(pdev->yImage * pdev->lDelta, 1);
    uint8_t *pInputImage  = pSurface;
    int      seed         = 0;
    int      x, y, p, pixel, threshold, error;
    int      width, height;
    int *    lerr;
    int *    cerr;
    int *    terr;
    uint8_t *bits, *new_bits;

    // Floyd & Steinberg error diffusion dithering
    // This algorithm use the following filter
    //          *   7
    //      3   5   1     (1/16)

    width  = pdev->lDelta;
    height = pdev->yImage;

    // allocate space for error arrays
    lerr = (int *)malloc(width * sizeof(int));
    cerr = (int *)malloc(width * sizeof(int));
    memset(lerr, 0, width * sizeof(int));
    memset(cerr, 0, width * sizeof(int));

    // left border
    error = 0;
    for (y = 0; y < height; y++) {
        bits     = pInputImage + (y * pdev->lDelta);
        new_bits = pOutputImage + (y * pdev->lDelta);

        threshold   = (WHITE / 2 + RAND(129) - 64);
        pixel       = bits[0] + error;
        p           = (pixel > threshold) ? WHITE : BLACK;
        error       = pixel - p;
        new_bits[0] = (uint8_t)p;
    }
    // right border
    error = 0;
    for (y = 0; y < height; y++) {
        bits     = pInputImage + (y * pdev->lDelta);
        new_bits = pOutputImage + (y * pdev->lDelta);

        threshold           = (WHITE / 2 + RAND(129) - 64);
        pixel               = bits[width - 1] + error;
        p                   = (pixel > threshold) ? WHITE : BLACK;
        error               = pixel - p;
        new_bits[width - 1] = (uint8_t)p;
    }
    // top border
    bits     = pInputImage;
    new_bits = pOutputImage;
    error    = 0;
    for (x = 0; x < width; x++) {
        threshold   = (WHITE / 2 + RAND(129) - 64);
        pixel       = bits[x] + error;
        p           = (pixel > threshold) ? WHITE : BLACK;
        error       = pixel - p;
        new_bits[x] = (uint8_t)p;
        lerr[x]     = INITERR(bits[x], p);
    }

    // interior bits
    for (y = 1; y < height; y++) {
        // scan left to right
        bits     = pInputImage + (y * pdev->lDelta);
        new_bits = pOutputImage + (y * pdev->lDelta);

        cerr[0] = INITERR(bits[0], new_bits[0]);
        for (x = 1; x < width - 1; x++) {
            error = (lerr[x - 1] + 5 * lerr[x] + 3 * lerr[x + 1] +
                     7 * cerr[x - 1]) /
                    16;
            pixel = bits[x] + error;
            if (pixel > (WHITE / 2)) {
                new_bits[x] = WHITE;
                cerr[x]     = pixel - WHITE;
            } else {
                new_bits[x] = BLACK;
                cerr[x]     = pixel - BLACK;
            }
        }
        // set errors for ends of the row
        cerr[0]         = INITERR(bits[0], new_bits[0]);
        cerr[width - 1] = INITERR(bits[width - 1], new_bits[width - 1]);

        // swap error buffers
        terr = lerr;
        lerr = cerr;
        cerr = terr;
    }

    free(lerr);
    free(cerr);
    // update the target with the resulting 8bpp image
    memcpy(pSurface, pOutputImage, pdev->lDelta * pdev->yImage);

    free(pOutputImage);
}

/******************************************************************************
 *  YMC_OutputStrip()
 *      Create 3 plane (YMC) 6 bit format data and save to memory.
 *
 * Parameters:
 *    pdev        Pointer to our PDEVICE
 *    pSurface    Pointer to the surface
 *
 *  Returns:
 *      None
 ****************************************************************************/
void YMC_OutputStrip(PDEVDATA pdev, uint8_t *pSurface) {
    uint32_t x        = 0;
    int32_t  y        = 0;
    uint8_t *lpSrc    = 0;
    uint8_t *lpPlane  = 0;
    uint32_t ulColour = 0;

    TRACE("pdev->lDelta:%u", pdev->lDelta);
    TRACE("pdev->xImage:%u", pdev->xImage);
    TRACE("pdev->yImage:%u", pdev->yImage);
    TRACE("pdev->ulCMYInPlaneWidth:%u", pdev->ulCMYInPlaneWidth);

    // portrait 1016 shown for 32bit surface
    for (y = 0; y < pdev->yImage; y++) {
        for (ulColour = 0; ulColour < 3; ulColour++) {
            lpSrc = pSurface + y * pdev->lDelta + (2 - ulColour);
            lpPlane =
                    pdev->lpPageBuffer[ulColour] + y * pdev->ulCMYInPlaneWidth;

            for (x = 0; x < pdev->ulCMYInPlaneWidth; x++) {
                *lpPlane++ = *lpSrc;
                lpSrc += 4;
            }
        }
    }
    return;
}

/*******************************************************************************
 *  HalftoneKStrip()
 *      Halftone 8bit gray to 1bit mono.
 *
 *  Parameters:
 *      pdev        Pointer to our PDEV
 *      pSurface    Pointer to our SURFOBJ
 *      y           scanline offset
 *
 *  Returns:
 *      None
 *******************************************************************************/

void HalftoneKStrip(PDEVDATA pdev, uint8_t *pSurface, int32_t y) {
    uint8_t *lpHalftoned = (uint8_t *)pdev->lpHalftone;
    uint8_t *lpPlane     = NULL;
    uint8_t *lpDst       = NULL;
    uint16_t cnt         = 0;
    int32_t  x           = 0;

    // ok so 8bit monochrome filled so convert to halftoned 1bpp
    lpPlane = (uint8_t *)pSurface + pdev->xImage * y;
    lpDst   = (uint8_t *)lpHalftoned;

    for (x = 0; x < pdev->xImage; x++) {
        uint8_t a;
        uint8_t b;
        b = ~(lpPlane[0]);
        a = 128 >> cnt;

        if (b) {
            *lpDst |= a;
        } else {
            *lpDst &= ~a;
        }

        if (cnt == 7) {
            lpDst++;
            cnt = 0;
        } else {
            cnt++;
        }
        lpPlane++;
    }
}

/*******************************************************************************
 *  CopyKPlaneToBuffer()
 *      Receive plane and write it on the buffer.
 *
 *  Parameters:
 *      pdev           Pointer to our PDEVICE
 *      lpSrc          Pointer to the image data.
 *                     If this is null, use pStripObj.
 *      pSurface       Pointer to the surface object.
 *      pSettings      Pointer to current settings.
 *      lYOffset       Current Y position.
 *
 *  Returns:
 *      None
 *******************************************************************************/

void CopyKPlaneToBuffer(PDEVDATA  pdev,
                        uint8_t * lpSrc,
                        uint8_t * pSurface,
                        PSETTINGS pSettings,
                        int32_t   lYOffset) {
    uint8_t *lpPlane      = NULL;
    bool     bDataInPlane = false;
    uint8_t *lpHalftoned  = NULL;
    uint32_t x            = 0;

    if (lpSrc != NULL) {

        /*******************************************************************************
         * Usually we have bitmap data in the pStripObj.
         * But when Detect Adjacent Colour option is set, we create it on
         *another buffer.
         *******************************************************************************/
        uint8_t *pBits = (uint8_t *)pSurface;

        if (PRO360_OEM(pSettings)) {
            memcpy(pBits, lpSrc, pdev->ulCMYInPlaneWidth);
        } else if (HELIX_OEM(pSettings)) {
            memcpy(pBits, lpSrc, pdev->ulCMYInPlaneWidth);
        } else {
            for (x = 0; x < pdev->ulCMYInPlaneWidth; x++) {
                pBits[x] = ~lpSrc[x];
            }
        }
    }

    // 8bpp monochrome

    lpPlane = pSurface;
    if (lpSrc == NULL) {
        lpPlane += lYOffset * pdev->lDelta; // Bitmap.pvData;
    }

    if (PRO360_OEM(pSettings)) {
        for (x = 0; x < pdev->ulCMYInPlaneWidth; x++) {
            if (*lpPlane != 0x00) {
                bDataInPlane = true;
                break;
            }
            lpPlane++;
        }
    } else if (HELIX_OEM(pSettings)) {
        for (x = 0; x < pdev->ulCMYInPlaneWidth; x++) {
            if (*lpPlane != 0x00) {
                bDataInPlane = true;
                break;
            }
            lpPlane++;
        }
    } else {
        for (x = 0; x < pdev->ulCMYInPlaneWidth; x++) {
            if (*lpPlane != 0xFF) {
                bDataInPlane = true;
                break;
            }
            lpPlane++;
        }
    }
    if (bDataInPlane) {
        HalftoneKStrip(pdev, pSurface, (lpSrc) ? 0 : lYOffset);
        lpHalftoned = (uint8_t *)pdev->lpHalftone;

        // Convert source strip into device format
        if (HELIX_OEM(pSettings) || PRO360_OEM(pSettings)) {
            CopyToBuffer(pdev,
                         (lpSrc == NULL) ? (uint8_t *)pSurface +
                                         (lYOffset * pdev->lDelta) :
                                 pSurface,
                         pdev->ulKOutPlaneWidth,
                         PLANE_K,
                         true);
        } else {
            format_printhead_data(pSettings,
                                  lpHalftoned,
                                  pdev->lpKOut,
                                  BITS_FOR_BLACK_DATA,
                                  pdev->ulKOutPlaneWidth);

            CopyToBuffer(
                    pdev, pdev->lpKOut, pdev->ulKOutPlaneWidth, PLANE_K, true);
        }
    } else {
        memset((uint8_t *)pSurface, 0, pdev->lDelta);
        CopyToBuffer(pdev, pSurface, pdev->ulKOutPlaneWidth, PLANE_K, false);
    }

    return;
}

/******************************************************************************
 *  K_OutputStrip()
 *      Create 1 bit (mono)  format data and save to memory.
 *
 *  Parameters:
 *      pdev         Pointer to our PDEVICE
 *      psettings
 *      pSurface     Pointer to the strip object
 *
 *  Returns:
 *      None
 ****************************************************************************/
void K_OutputStrip(PDEVDATA pdev, PSETTINGS pSettings, uint8_t *pSurface) {
    int32_t y = 0;

    error_diffusion(pdev, pSurface);

    for (y = 0; y < pdev->yImage; y++) {
        CopyKPlaneToBuffer(pdev, NULL, pSurface, pSettings, y);
    }

    return;
}

/******************************************************************************
 *  YMCK_OutputStrip()
 *      Create 3 plane (YMC) 6 bit + 1 bit (mono)  format data and save to
 *      memory.
 *
 *  Parameters:
 *      pdev         Pointer to our PDEVICE
 *      pSettings    Pointers to our Settings
 *      pSurface     Pointer to the strip object
 *
 *  Returns:
 *      None
 ****************************************************************************/

void YMCK_OutputStrip(PDEVDATA pdev, PSETTINGS pSettings, uint8_t *pSurface) {
    uint32_t x          = 0;
    int32_t  y          = 0;
    uint8_t *pKSrcStrip = pdev->lpKSrcStrip;
    uint8_t *lpPlane    = NULL;
    uint8_t *lpSrc      = NULL;
    uint32_t ulColour   = 0;

    // apply halftoning to K-resin plane if monobitmaps have been requested to
    // be printed on the kresin plane..
    // not supported currently
    TRACE("pdev->xImage: %u, pdev->epPaperXdots: %u",
          pdev->xImage,
          pdev->epPaperXdots);
    TRACE("pdev->ulCMYInPlaneWidth: %u, pdev->lDelta: %u",
          pdev->ulCMYInPlaneWidth,
          pdev->lDelta);

    for (y = 0; y < pdev->yImage; y++) {

        // Check whether the "Detect adjacent colour" option is set for this
        // CMYK page
        if (pdev->bDetectAdjacentColour) {
            // NB bDetectAdjacentColour is ALWAYS set!!
            // Save off the strip into four page buffers (one for each colour
            // plane).
            // These planes will be processed via DetectAdjacentColour() at
            // dvEndPage().
            // Note that CopyToBuffer() is called as the final step in
            // DetectAdjacentColour()
            // so it is bypassed at this point.

            lpSrc   = (uint8_t *)pSurface + (y * pdev->lDelta) + 3;
            lpPlane = pdev->lpPageBuffer[PLANE_K] + y * pdev->ulCMYInPlaneWidth;

            for (x = 0; x < pdev->ulCMYInPlaneWidth; x++) {
                *lpPlane++ = *lpSrc;
                lpSrc += 4;
            }

            // Copy colour plane from STRIPOBJ.
            for (ulColour = 0; ulColour < 3; ulColour++) {
                lpSrc = (uint8_t *)pSurface + y * pdev->lDelta + (2 - ulColour);
                lpPlane = pdev->lpPageBuffer[ulColour] +
                          y * pdev->ulCMYInPlaneWidth;

                for (x = 0; x < pdev->ulCMYInPlaneWidth; x++) {
                    *lpPlane++ = *lpSrc;
                    lpSrc += 4;
                }
            }
        } else {
            // this is never called
            CopyKPlaneToBuffer(pdev, NULL, pKSrcStrip, pSettings, y);

            for (ulColour = 0; ulColour < 3; ulColour++) {
                lpSrc = (uint8_t *)pSurface + y * pdev->lDelta + (2 - ulColour);
                CopyColourPlaneToBuffer(
                        pdev, pSettings, lpSrc, ulColour, false);
            }
        }
    }
    return;
}

/******************************************************************************
 *  dvOutputStrip()
 *      Send a strip of raster data to the printer
 *
 *  Pararneters:
 *      pdev        Pointer to our PDEV
 *      pSettings
 *      pSurface    Pointer to the output surface
 *
 *  Returns:
 *      None
 ****************************************************************************/
void dvOutputStrip(PDEVDATA pdev, PSETTINGS pSettings, uint8_t *pSurface) {
    if (IgnoreImageData(pSettings)) {
        return;
    }

    switch (pdev->eChannelOption) {
        case UICBVAL_YMC:
            YMC_OutputStrip(pdev, pSurface);
            break;

        case UICBVAL_YMCK:
        default:
            YMCK_OutputStrip(pdev, pSettings, pSurface);
            break;

        case UICBVAL_KResin:
            K_OutputStrip(pdev, pSettings, pSurface);
            break;
    }

    return;
}

void ColourCorrect(CHARPIXEL *pixel, uint8_t intent) {
    // call this function as a LUT algorithm in the same way Naive is an
    // algorithm.

    //  Algorithm based on :    (Red - Red_i)/(Red_i+1 -Red_i) = (Cyan-
    //  Cyan_i)/(Cyan_i+1 - Cyan_i) where i goes from 0 to 255 in steps of 4

    unsigned char upper[3];
    unsigned char lower[3];
    unsigned char offset[3];

    unsigned char r = 0; // just a convenience
    unsigned char g = 1;
    unsigned char b = 2;

    lower[r]  = (unsigned char)(pixel->r) / GridDelta;
    upper[r]  = lower[r] + 1;
    offset[r] = (unsigned char)(pixel->r - lower[r] * GridDelta);
    lower[g]  = (unsigned char)(pixel->g) / GridDelta;
    upper[g]  = lower[g] + 1;
    offset[g] = (unsigned char)(pixel->g - lower[g] * GridDelta);
    lower[b]  = (unsigned char)(pixel->b) / GridDelta;
    upper[b]  = lower[b] + 1;
    offset[b] = (unsigned char)(pixel->b - lower[b] * GridDelta);

    pixel->c = Cyan[lower[r]][lower[g]][lower[b]][intent];
    pixel->c += (unsigned char)((offset[r]) ? (1.0 * offset[r] *
                                               (Cyan[upper[r]][upper[g]]
                                                    [upper[b]][intent] -
                                                Cyan[lower[r]][lower[g]]
                                                    [lower[b]][intent])) /
                                                GridDelta :
                                        0);

    pixel->m = Magenta[lower[r]][lower[g]][lower[b]][intent];
    pixel->m += (unsigned char)((offset[g]) ? (1.0 * offset[g] *
                                               (Magenta[upper[r]][upper[g]]
                                                       [upper[b]][intent] -
                                                Magenta[lower[r]][lower[g]]
                                                       [lower[b]][intent])) /
                                                GridDelta :
                                        0);

    pixel->y = Yellow[lower[r]][lower[g]][lower[b]][intent];
    pixel->y += (unsigned char)((offset[b]) ? (1.0 * offset[b] *
                                               (Yellow[upper[r]][upper[g]]
                                                      [upper[b]][intent] -
                                                Yellow[lower[r]][lower[g]]
                                                      [lower[b]][intent])) /
                                                GridDelta :
                                        0);
}

void ColourCorrectHelix(CHARPIXEL *pixel, uint8_t intent) {
    unsigned char upper[3];
    unsigned char lower[3];
    unsigned char offset[3];

    double adjustment = 0;
    int    adj        = 0;

    unsigned char r = 0; // just a convenience
    unsigned char g = 1;
    unsigned char b = 2;

    lower[r]  = (unsigned char)(pixel->r) / GridDelta;
    upper[r]  = lower[r] + 1;
    offset[r] = (unsigned char)(pixel->r - lower[r] * GridDelta);
    lower[g]  = (unsigned char)(pixel->g) / GridDelta;
    upper[g]  = lower[g] + 1;
    offset[g] = (unsigned char)(pixel->g - lower[g] * GridDelta);
    lower[b]  = (unsigned char)(pixel->b) / GridDelta;
    upper[b]  = lower[b] + 1;
    offset[b] = (unsigned char)(pixel->b - lower[b] * GridDelta);

    pixel->c   = Cyan[lower[r]][lower[g]][lower[b]][intent];
    adjustment = 1.0 * offset[r] *
                 (Cyan[upper[r]][upper[g]][upper[b]][intent] -
                  Cyan[lower[r]][lower[g]][lower[b]][intent]) /
                 GridDelta;
    adj = adjustment < 0.0 ? (int)ceil(adjustment - 0.5) :
                             (int)floor(adjustment + (double)0.5);
    pixel->c += (unsigned char)adj;

    pixel->m   = Magenta[lower[r]][lower[g]][lower[b]][intent];
    adjustment = 1.0 * offset[g] *
                 (Magenta[upper[r]][upper[g]][upper[b]][intent] -
                  Magenta[lower[r]][lower[g]][lower[b]][intent]) /
                 GridDelta;
    adj = adjustment < 0.0 ? (int)ceil(adjustment - 0.5) :
                             (int)floor(adjustment + 0.5);
    pixel->m += (unsigned char)adj;

    pixel->y   = Yellow[lower[r]][lower[g]][lower[b]][intent];
    adjustment = 1.0 * offset[b] *
                 (Yellow[upper[r]][upper[g]][upper[b]][intent] -
                  Yellow[lower[r]][lower[g]][lower[b]][intent]) /
                 GridDelta;
    adj = adjustment < 0.0 ? (int)ceil(adjustment - 0.5) :
                             (int)floor(adjustment + 0.5);
    pixel->y += (unsigned char)adj;
}

bool SendPage(PDEVDATA            pdev,
              PSETTINGS           psettings,
              cups_page_header2_t header,
              cups_raster_t *     ras,
              uint8_t *           rasterData) {
    // target 32bpp/8bpp surface
    uint8_t *pSurface = NULL;
    uint32_t x, y;

    // firmware job so bail
    if (pdev->bFWDownload) {
        return true;
    }

    // initialisation of layers and counters

    // initialise whether there is a Cyan layer to send
    pdev->epbCyanLayer = false;
    // initialise whether there is a Magenta layer to send
    pdev->epbMagentaLayer = false;
    // initialise whether there is a Yellow layer to send
    pdev->epbYellowLayer = false;
    // initialise whether there is a black layer to send
    pdev->epbBlackLayer = false;

    // initialise size of the Cyan layer data
    pdev->eplCyanImageDataLen = 0L;
    // initialise size of the Magenta layer data
    pdev->eplMagentaImageDataLen = 0L;
    // initialise size of the Yellow layer
    pdev->eplYellowImageDataLen = 0L;
    // initialise size of the black layer data
    pdev->eplBlackImageDataLen = 0L;

    if (pdev->pso32bitSurface) {
        uint8_t *lpSrc;
        uint8_t *lpDst;
        uint32_t srcColour;
        uint32_t cmyk;
        uint32_t NextPixel;

        bool              bColourMatch = false;
        PCOLOURMATCHTABLE pct          = NULL;
        TRACE("pdev->epOffsetX : %d", pdev->epOffsetX);

        if (pdev->eChannelOption == UICBVAL_KResin) {
            pSurface     = pdev->lpKPSrcStrip;
            NextPixel    = 1;
            pdev->lDelta = pdev->xImage;
            TRACE("UICBVAL_KResin : pdev->lDelta : %d", pdev->lDelta);

        } else {
            pSurface  = pdev->pso32bitSurface;
            NextPixel = 4;
            // PORTRAIT!!? 642 * 4
            pdev->lDelta = pdev->xImage * 4;
        }
        TRACE("lDelta %d", pdev->lDelta);

        if ((PRONTO_TYPE(psettings) || ENDURO_TYPE(psettings) ||
             ENDURO_PLUS_TYPE(psettings) || RIOPRO_TYPE(psettings) ||
             XXL_TYPE(psettings)) &&
            (psettings->nColourCorrection >=
             UICBVAL_ColourCorrection_ICC_Internal)) {
            bColourMatch = true;
            TRACE("CATULTRA bCOLORMATCH ENABLED %d",
                  psettings->nColourCorrection);
            }

        // move 24bpp data into our 32bpp/8bpp surface
        if (psettings->OEM == OEM_PRO360) {
            int32_t ycnt, xcnt;

            // Convert the value of nColourCorrection to the
            // UICBVAL_ColourCorrection_* value to make the code simpler to
            // follow
            int32_t cc =
                    ConvertColourCorrectionValue(psettings->nColourCorrection);

            // Calculate the intent, (offset) into the colour correction tables,
            // i.e. calculate the correct one to use.
            uint8_t intent = 0;
            if (cc >= UICBVAL_ColourCorrection_Perceptual) {
                intent = cc - UICBVAL_ColourCorrection_Perceptual;
            }

            bool isInternalColourCorrection =
                    cc == UICBVAL_ColourCorrection_ICC_Internal;

            // Used when colour correction is set to
            // UICBVAL_ColourCorrection_ICC_Internal
            pct = (PCOLOURMATCHTABLE)&ColorMatchTableRioPro;

            // Determine whether colour correction needs to be done at all.
            bool applyColourCorrection =
                    isInternalColourCorrection ||
                    (cc >= UICBVAL_ColourCorrection_Perceptual &&
                     cc <= UICBVAL_ColourCorrection_AbsColorimetric);

            TRACE("applyColourCorrection = %s, psettings->nColourCorrection = "
                  "%d, isInternalColourCorrection = %s",
                  ShowTrueFalse(applyColourCorrection),
                  psettings->nColourCorrection,
                  ShowTrueFalse(isInternalColourCorrection));

            TRACE("cc = %d, name = %s, intent = %s (%d)",
                  cc,
                  DecodeColourCorrection(cc),
                  DecodeIntent(intent),
                  intent);

            TRACE("bColourMatch = %s", ShowTrueFalse(bColourMatch));

            TRACE("pdev->epPaperXdots : %d", pdev->epPaperXdots);
            TRACE("pdev->epPaperYdots : %d", pdev->epPaperYdots);
            TRACE("psettings->orientation : %d", psettings->orientation);
            TRACE("pdev->lDelta : %d", pdev->lDelta);

            if (psettings->orientation == 0) {
                xcnt = pdev->epPaperYdots;
                ycnt = (pdev->epPaperXdots - pdev->epOffsetX);
            } else {
                xcnt = (pdev->epPaperXdots);
                ycnt = (pdev->epPaperYdots);
            }

            TRACE("passed SendPage header.cupsHeight %d", header.cupsHeight);
            TRACE("passed SendPage header.cupsBytesPerLine %d",
                  header.cupsBytesPerLine);
            TRACE("360 passed SendPage header.Orientation %d",
                  header.Orientation);
            TRACE("600 passed SendPage header.Orientation %d",
                  header.Orientation);
            TRACE("epPaperXdots %d", pdev->epPaperXdots);
            TRACE("epPaperYdots %d", pdev->epPaperYdots);

            for (y = 0; y < header.cupsHeight; y++) {
                if ((y % 50) == 0) {
                    TRACE("y = %d, header.cupsHeight = %d",
                          y,
                          header.cupsHeight);
                }

                if (CUPSRASTERREADPIXELS(
                            ras, rasterData, header.cupsBytesPerLine) < 1) {
                    TRACE("breaking out !! %d", y);
                    break;
                }
                lpSrc = rasterData;

                for (x = 0; x < (header.cupsBytesPerLine / 3); x++) {
                    //           1013                               642
                    //    --------------------------        ----------------
                    //    -oooo                             -
                    //    -                                 -
                    //    -                                 -
                    // 642 -                                 -
                    //    -                                 -
                    //    -                                 -
                    //    -                                 -
                    //                                      -
                    //                                      -
                    //                                      -o

                    // landscape
                    if (psettings->orientation == 0) {
                        if (psettings->xdpi == 600) {
                            if (pdev->eChannelOption == UICBVAL_KResin) {
                                if (pdev->bRotate) {
                                    lpDst = (uint8_t *)pSurface +
                                            (((xcnt - 1) - x) * pdev->lDelta) +
                                            (((ycnt - 1) - (y / 2)) *
                                             NextPixel);
                                } else {
                                    lpDst = (uint8_t *)pSurface +
                                            (x * pdev->lDelta) +
                                            ((y / 2) * NextPixel);
                                }
                            } else {
                                if (pdev->bRotate) {
                                    lpDst = (uint8_t *)pSurface +
                                            (((xcnt - 1) - (x / 2)) *
                                             pdev->lDelta) +
                                            (((ycnt - 1) - (y / 2)) *
                                             NextPixel);
                                } else {
                                    lpDst = (uint8_t *)pSurface +
                                            ((x / 2) * pdev->lDelta) +
                                            ((y / 2) * NextPixel);
                                }
                            }
                        } else {
                            if (pdev->bRotate) {
                                lpDst = (uint8_t *)pSurface +
                                        (((xcnt - 1) - x) * pdev->lDelta) +
                                        (((ycnt - 1) - y) * NextPixel);
                            } else {
                                lpDst = (uint8_t *)pSurface +
                                        (x * pdev->lDelta) + (y * NextPixel);
                            }
                        }
                    } else {
                        if (psettings->xdpi == 600) {
                            if (pdev->eChannelOption == UICBVAL_KResin) {
                                if (pdev->bRotate) { // 180 flip
                                    lpDst = (uint8_t *)pSurface +
                                            ((y)*pdev->lDelta) +
                                            (((xcnt - 1) - (x / 2)) *
                                             NextPixel);
                                } else {
                                    lpDst = (uint8_t *)pSurface +
                                            (((ycnt - 1) - (y)) *
                                             pdev->lDelta) +
                                            ((x / 2) * NextPixel);
                                }
                            } else {
                                if (pdev->bRotate) { // 180 flip
                                    lpDst = (uint8_t *)pSurface +
                                            ((y / 2) * pdev->lDelta) +
                                            (((xcnt - 1) - (x / 2)) *
                                             NextPixel);
                                } else {
                                    int32_t yoffset = (ycnt - 1) - y;
                                    yoffset         = yoffset >> 1;
                                    yoffset *= pdev->lDelta;
                                    lpDst = (uint8_t *)pSurface + yoffset +
                                            ((x / 2) * NextPixel);
                                }
                            }
                        } else {
                            if (pdev->bRotate) { // 180 flip
                                lpDst = (uint8_t *)pSurface +
                                        (y * pdev->lDelta) +
                                        (((xcnt - 1) - x) * NextPixel);
                            } else {
                                lpDst = (uint8_t *)pSurface +
                                        (((ycnt - 1) - y) * pdev->lDelta) +
                                        (x * NextPixel);
                            }
                        }
                    }
                    // set RGB pixels into 32bpp/8bpp plane
                    srcColour = RGB(lpSrc[(x * 3)],
                                    lpSrc[(x * 3) + 1],
                                    lpSrc[(x * 3) + 2]);

                    if (pdev->eBlackStartOption == UICBVAL_Start_YMC ||
                        pdev->eChannelOption == UICBVAL_YMC) {
                        cmyk = RGB_to_CMY((uint32_t *)&srcColour);
                    } else {
                        cmyk = RGB_to_CMYK((uint32_t *)&srcColour);
                    }

                    if (pdev->eChannelOption == UICBVAL_KResin) {
                        *lpDst = ~RgbToGray(lpSrc[x * 3],
                                            lpSrc[(x * 3) + 1],
                                            lpSrc[(x * 3) + 2]);
                    } else {
                        // built in colormatching requested?
                        if (applyColourCorrection && cmyk != (int32_t)0xff) {

                            if (isInternalColourCorrection) {
                                rgb in;

                                in.r = lpSrc[x * 3];
                                in.g = lpSrc[(x * 3) + 1];
                                in.b = lpSrc[(x * 3) + 2];

                                hsv out = rgb2hsv(in);
                                cmyk = hsvToCMY((uint16_t)out.h,
                                                (uint16_t)out.s,
                                                (uint16_t)out.v,
                                                pct);

                            } else {

                                CHARPIXEL cp;
                                cp.r = lpSrc[x * 3];
                                cp.g = lpSrc[(x * 3) + 1];
                                cp.b = lpSrc[(x * 3) + 2];

                                ColourCorrectHelix(&cp, intent);

                                cmyk = ((uint32_t)(((uint8_t)(cp.c) << 24 |
                                                    ((uint8_t)(cp.m) << 16)) |
                                                   ((uint8_t)(cp.y) << 8)) |
                                        ((uint8_t)(0)));
                            }
                        }

                        if (psettings->xdpi == 600) {
                            uint8_t *lpDst2;
                            // pSurface600K = pdev->lpKSrcStrip;

                            if (psettings->orientation == 0) {
                                if (pdev->bRotate) {
                                    lpDst2 = (uint8_t *)pSurface +
                                             ((((xcnt - 1) - x)) *
                                              pdev->lDelta) +
                                             (((ycnt - 1) - (y / 2)) *
                                              NextPixel);
                                } else {
                                    lpDst2 = (uint8_t *)pSurface +
                                             ((y)*pdev->lDelta) +
                                             (((xcnt - 1) - (x)) * NextPixel);
                                }
                            } else {
                                if (pdev->bRotate) { // 180 flip
                                    lpDst2 = (uint8_t *)pSurface +
                                             (y * pdev->lDelta) +
                                             (((xcnt - 1) - (x >> 1)) *
                                              NextPixel);
                                } else {
                                    lpDst2 = (uint8_t *)pSurface +
                                             (((ycnt - 1) - (y)) *
                                              pdev->lDelta) +
                                             ((x >> 1) * NextPixel);
                                }
                            }

                            *lpDst        = GetCValue(cmyk);
                            *(lpDst + 1)  = GetMValue(cmyk);
                            *(lpDst + 2)  = GetYValue(cmyk);
                            *(lpDst2 + 3) = GetKValue(cmyk);
                        } else {
                            *lpDst       = GetCValue(cmyk);
                            *(lpDst + 1) = GetMValue(cmyk);
                            *(lpDst + 2) = GetYValue(cmyk);
                            *(lpDst + 3) = GetKValue(cmyk);
                        }
                    }
                }
            }
            TRACE_STR("360 passed SendPage Complete");
        } else if (psettings->OEM == OEM_HELIX) {

            TRACE("psettings->nColourCorrection %d",
                  psettings->nColourCorrection);

            int32_t ycnt, xcnt;
            TRACE("pdev->epPaperXdots : %d", pdev->epPaperXdots);
            TRACE("pdev->epPaperYdots : %d", pdev->epPaperYdots);

            if (psettings->orientation == 0) {
                xcnt = pdev->epPaperYdots;
                ycnt = (pdev->epPaperXdots - pdev->epOffsetX);
            } else {
                xcnt = (pdev->epPaperXdots);
                ycnt = (pdev->epPaperYdots);
            }

            TRACE("passed SendPage header.cupsHeight %d", header.cupsHeight);
            TRACE("passed SendPage header.cupsBytesPerLine %d",
                  header.cupsBytesPerLine);
            TRACE("passed SendPage header.Orientation %d", header.Orientation);

            // Apply colour correction if psettings->nColourCorrection is not
            // UICBVAL_HELIX_ColourCorrection_None or
            // UICBVAL_HELIX_ColourCorrection_ICC_External
            bool applyColourCorrection =
                    (psettings->nColourCorrection >=
                     UICBVAL_HELIX_ColourCorrection_Perceptual) &&
                    (psettings->nColourCorrection <
                     UICBVAL_HELIX_ColourCorrection_ICC_External);

            uint8_t intent = psettings->nColourCorrection -
                             UICBVAL_HELIX_ColourCorrection_Perceptual;

            TRACE("psettings->nColourCorrection = %s (%d), "
                  "applyColourCorrection = %s, intent = %d",
                  DecodeHelixColourCorrection(psettings->nColourCorrection),
                  psettings->nColourCorrection,
                  ShowTrueFalse(applyColourCorrection),
                  intent);

            TRACE("header.cupsBytesPerLine = %d, header.cupsHeight = %d",
                  header.cupsBytesPerLine,
                  header.cupsHeight);

            for (y = 0; y < header.cupsHeight; y++) {
                if (CUPSRASTERREADPIXELS(
                            ras, rasterData, header.cupsBytesPerLine) < 1) {
                    TRACE("breaking out !! %u", y);
                    break;
                }

                lpSrc = rasterData;

                for (x = 0; x < (header.cupsBytesPerLine / 3); x++) {
                    //           1036                               664
                    //    --------------------------        ----------------
                    //    -oooo                             -
                    //    -                                 -
                    //    -                                 -
                    // 648 -                                 -
                    //    -                                 -
                    //    -                                 -
                    //    -                                 -
                    //                                      -
                    //                                      -
                    //                                      -o

                    // landscape
                    if (psettings->orientation == 0) {
                        if (pdev->bRotate) {
                            lpDst = (uint8_t *)pSurface +
                                    (((xcnt - 1) - x) * pdev->lDelta) +
                                    (((ycnt - 1) - y) * NextPixel);
                        } else {
                            lpDst = (uint8_t *)pSurface + (x * pdev->lDelta) +
                                    (y * NextPixel);
                        }
                    } else {
                        if (pdev->bRotate) { // 180 flip
                            lpDst = (uint8_t *)pSurface + (y * pdev->lDelta) +
                                    (((xcnt - 1) - x) * NextPixel);
                        } else {
                            lpDst = (uint8_t *)pSurface +
                                    (((ycnt - 1) - y) * pdev->lDelta) +
                                    (x * NextPixel);
                        }
                    }
                    // set RGB pixels into 32bpp/8bpp plane
                    srcColour = RGB(lpSrc[x * 3],
                                    lpSrc[(x * 3) + 1],
                                    lpSrc[(x * 3) + 2]);

                    if (pdev->eBlackStartOption == UICBVAL_Start_YMC ||
                        pdev->eChannelOption == UICBVAL_YMC) {
                        cmyk = RGB_to_CMY(&srcColour);
                    } else {
                        cmyk = RGB_to_CMYK((uint32_t *)&srcColour);
                    }

                    if (pdev->eChannelOption == UICBVAL_KResin) {
                        *lpDst = ~RgbToGray(lpSrc[x * 3],
                                            lpSrc[(x * 3) + 1],
                                            lpSrc[(x * 3) + 2]);
                    } else {
                        if (applyColourCorrection && cmyk != 0xff &&
                            cmyk != 0) {
                            CHARPIXEL cp;
                            cp.r = lpSrc[x * 3];
                            cp.g = lpSrc[(x * 3) + 1];
                            cp.b = lpSrc[(x * 3) + 2];

                            // Calculate the 'intent' based on the colour
                            // correction selected
                            ColourCorrectHelix(&cp, intent);

                            cmyk = ((uint32_t)(((uint8_t)(cp.c) << 24 |
                                                ((uint8_t)(cp.m) << 16)) |
                                               ((uint8_t)(cp.y) << 8)) |
                                    ((uint8_t)(0)));
                        }

                        *lpDst       = GetCValue(cmyk);
                        *(lpDst + 1) = GetMValue(cmyk);
                        *(lpDst + 2) = GetYValue(cmyk);
                        *(lpDst + 3) = GetKValue(cmyk);
                    }
                }
            }
        } else {
            int32_t ycnt, xcnt;
            TRACE("pdev->epPaperXdots: %x", pdev->epPaperXdots);
            TRACE("pdev->epPaperYdots: %x", pdev->epPaperYdots);

            if (psettings->orientation == 0) {
                ycnt = pdev->epPaperYdots;
                xcnt = (pdev->epPaperXdots - pdev->epOffsetX);
            } else {
                ycnt = (pdev->epPaperXdots);
                xcnt = (pdev->epPaperYdots);
            }

            TRACE("passed SendPage header.cupsHeight %d", header.cupsHeight);
            TRACE("passed SendPage header.cupsBytesPerLine %d",
                  header.cupsBytesPerLine);
            TRACE("passed SendPage header.Orientation %d", header.Orientation);
            TRACE("psettings->nColourCorrection %d",
                  psettings->nColourCorrection);
            for (y = 0; y < header.cupsHeight; y++) {
                if (CUPSRASTERREADPIXELS(
                            ras, rasterData, header.cupsBytesPerLine) < 1) {
                    TRACE("breaking out at lin %d", y);
                    break;
                }
                lpSrc = rasterData;

                for (x = 0; x < (header.cupsBytesPerLine / 3); x++) {
                    //           1016                               648
                    //    --------------------------        ----------------
                    //    -oooo                             -
                    //    -                                 -
                    //    -                                 -
                    // 648 -                                 -
                    //    -                                 -
                    //    -                                 -
                    //    -                                 -
                    //                                      -
                    //                                      -
                    //                                      -o

                    // landscape
                    if (psettings->orientation == 0) {
                        if (pdev->bRotate) {
                            lpDst = (uint8_t *)pSurface +
                                    (((xcnt - 1) - x) * pdev->lDelta) +
                                    ((y)*NextPixel);
                        } else {
                            lpDst = (uint8_t *)pSurface + ((x)*pdev->lDelta) +
                                    (((ycnt - 1) - y) * NextPixel);
                        }
                    } else {
                        if (pdev->bRotate) { // 180 flip
                            lpDst = (uint8_t *)pSurface +
                                    (((ycnt - 1) - y) * pdev->lDelta) +
                                    (((xcnt - 1) - x) * NextPixel);
                        } else {
                            lpDst = (uint8_t *)pSurface + (y * pdev->lDelta) +
                                    ((x + pdev->epOffsetX) * NextPixel);
                        }
                    }
                    // set RGB pixels into 32bpp/8bpp plane
                    srcColour = RGB(lpSrc[x * 3],
                                    lpSrc[(x * 3) + 1],
                                    lpSrc[(x * 3) + 2]);

                    if (pdev->eBlackStartOption == UICBVAL_Start_YMC ||
                        pdev->eChannelOption == UICBVAL_YMC) {
                        cmyk = RGB_to_CMY(&srcColour);
                    } else {
                        cmyk = RGB_to_CMYK((uint32_t *)&srcColour);
                    }

                    if (pdev->eChannelOption == UICBVAL_KResin) {
                        *lpDst = RgbToGray(lpSrc[x * 3],
                                           lpSrc[(x * 3) + 1],
                                           lpSrc[(x * 3) + 2]);
                    } else {
                        static CHARPIXEL cp;

                        // built in colormatching requested?
                        if (bColourMatch && cmyk != (int32_t)0xff) {
                            cp.r = lpSrc[x * 3];
                            cp.g = lpSrc[(x * 3) + 1];
                            cp.b = lpSrc[(x * 3) + 2];

                            if (psettings->nColourCorrection ==
                                UICBVAL_ColourCorrection_ICC_Internal) {
                                ColourCorrect(&cp, 4);
                            } else {
                                ColourCorrect(&cp,
                                              psettings->nColourCorrection - 2);
                            }
                            cmyk = ((uint32_t)(((uint8_t)(cp.c) << 24 |
                                                ((uint8_t)(cp.m) << 16)) |
                                               ((uint8_t)(cp.y) << 8)) |
                                    ((uint8_t)(0)));
                        }

                        *lpDst       = GetCValue(cmyk) >> 2;
                        *(lpDst + 1) = GetMValue(cmyk) >> 2;
                        *(lpDst + 2) = GetYValue(cmyk) >> 2;
                        *(lpDst + 3) = GetKValue(cmyk) >> 2;
                    }
                }
            } // End for
            }
        }

    TRACE_STR("About to call dvOutputStrip()");

    dvOutputStrip(pdev, psettings, pSurface);

    return true;
}

/****************************************************************************
 *
 *  DecStringToUint32()
 *      Converts a decimal string to a uint32_t value.
 *
 *  Type:
 *      Global function.
 *
 *  Parameters:
 *      chDecString      String containing decimal data.
 *      pdwDecValue      Pointer to uint32_t that will return value.
 *
 *  Returns:
 *      Normally true.  However if the function is passed a string of
 *      zero length or contains non-decimal data, it will abort and
 *      return false.
 *
 ****************************************************************************/
bool DecimalStrToUint32(uint8_t *chDecString, uint32_t *pdwDecValue) {
    int      i;
    int      iNumChars = (int)strlen((char *)chDecString);
    uint32_t dwTemp    = 0;
    bool     fReturn   = false;

    if (iNumChars != 0) {
        // Loop through the string and construct the number
        for (i = 0; i < iNumChars; i++) {
            int iDigit     = 0;
            int iPowerOf10 = (int)pow((double)10, (double)iNumChars - i - 1);

            if (*(chDecString + i) >= '0' && *(chDecString + i) <= '9') {
                iDigit = *(chDecString + i) - '0';
            } else {
                return fReturn;
            }

            dwTemp += iDigit * iPowerOf10;
        }

        // We have constructed a number! Set return value to true
        *pdwDecValue = dwTemp;
        fReturn      = true;
    }

    return fReturn;
}

/****************************************************************************
 *
 *  ParseEnduroStatus()
 *     Called to parse the Enduro Status string.
 *
 *  Parameters:
 *      pLMInst           Language Monitor Instance data
 *      pEnduroStatusData Pointer to buffer containing data from Printer
 *
 *  Returns:
 *      None.
 *
 ****************************************************************************/
void ParseEnduroStatus(SETTINGS *pSettings, char *pEnduroStatusData) {
    PENDURO_STATUS pEStatus = &pSettings->es;
    uint8_t *      pszData  = (uint8_t *)pEnduroStatusData;
    uint8_t        DataID   = 0;

    pEStatus->bPrinterConnected = true;

    // Walk the string. When one of the separators is met, then we change this
    // to a NULL. We then process the current data and reset the pointers to
    // the start of the next data entry
    while (*pEnduroStatusData != 0) {
        switch (*pEnduroStatusData) {
            case ':':
            case ',':
            case 0x03:
                // Change the ',' to a NULL to terminate the data section
                *pEnduroStatusData++ = 0;

                // Process the data strings
                switch (DataID) {
                    case 0:
                        DecimalStrToUint32(pszData, &pEStatus->eModel);
                        break;
                    case 1:
                        mc_strncpy(pEStatus->sModel,
                                   (char *)pszData,
                                   sizeof(pEStatus->sModel));
                        break;
                    case 2:
                        DecimalStrToUint32(pszData, &pEStatus->ePrintheadType);
                        break;
                    case 3:
                        mc_strncpy(pEStatus->sPrinterSerial,
                                   (char *)pszData,
                                   sizeof(pEStatus->sPrinterSerial));
                        break;
                    case 4:
                        mc_strncpy(pEStatus->sPrintheadSerial,
                                   (char *)pszData,
                                   sizeof(pEStatus->sPrintheadSerial));
                        break;
                    case 5:
                        mc_strncpy(pEStatus->sPCBSerial,
                                   (char *)pszData,
                                   sizeof(pEStatus->sPCBSerial));
                        break;
                    case 6:
                        mc_strncpy(pEStatus->sFirmwareVersion,
                                   (char *)pszData,
                                   sizeof(pEStatus->sFirmwareVersion));
                        break;
                    case 7:
                        if (*pszData != '3') {
                            pEStatus->iES_Density = DEFAULT_RIOPRO_DENSITY;
                        } else {
                            pszData += 2; // Move to density value
                            DecimalStrToUint32(pszData, &pEStatus->iES_Density);
                        }
                        break;
                    case 8:
                        DecimalStrToUint32(pszData, &pEStatus->iHandFeed);
                        break;
                    case 9:
                        DecimalStrToUint32(pszData, &pEStatus->iCardsPrinted);
                        break;
                    case 10:
                        DecimalStrToUint32(pszData,
                                           &pEStatus->iCardsOnPrinthead);
                        break;
                    case 11:
                        DecimalStrToUint32(pszData,
                                           &pEStatus->iDyePanelsPrinted);
                        break;
                    case 12:
                        DecimalStrToUint32(pszData,
                                           &pEStatus->iCleansSinceShipped);
                        break;
                    case 13:
                        DecimalStrToUint32(pszData,
                                           &pEStatus->iDyePanelsSinceClean);
                        break;
                    case 14:
                        DecimalStrToUint32(pszData,
                                           &pEStatus->iCardsSinceClean);
                        break;
                    case 15:
                        DecimalStrToUint32(pszData,
                                           &pEStatus->iCardsBetweenCleans);
                        break;
                    case 16:
                        DecimalStrToUint32(pszData, &pEStatus->iPrintHeadPosn);
                        break;
                    case 17:
                        DecimalStrToUint32(pszData, &pEStatus->iImageStartPosn);
                        break;
                    case 18:
                        DecimalStrToUint32(pszData, &pEStatus->iImageEndPosn);
                        break;
                    case 19:
                        DecimalStrToUint32(pszData, &pEStatus->iMajorError);
                        break;
                    case 20:
                        DecimalStrToUint32(pszData, &pEStatus->iMinorError);
                        break;
                    case 21:
                        mc_strncpy(pEStatus->sTagUID,
                                   (char *)pszData,
                                   sizeof(pEStatus->sTagUID));
                        break;
                    case 22:
                        DecimalStrToUint32(pszData, &pEStatus->iShotsOnFilm);
                        break;
                    case 23:
                        DecimalStrToUint32(pszData, &pEStatus->iShotsUsed);
                        break;
                    case 24:
                        mc_strncpy(pEStatus->sDyeFilmType,
                                   (char *)pszData,
                                   sizeof(pEStatus->sDyeFilmType));
                        break;
                    case 25:
                        DecimalStrToUint32(pszData, &pEStatus->iColourLength);
                        break;
                    case 26:
                        DecimalStrToUint32(pszData, &pEStatus->iResinLength);
                        break;
                    case 27:
                        DecimalStrToUint32(pszData, &pEStatus->iOvercoatLength);
                        break;
                    case 28:
                        DecimalStrToUint32(pszData, &pEStatus->eDyeFlags);
                        break;
                    case 29:
                        DecimalStrToUint32(pszData, &pEStatus->iCommandCode);
                        break;
                    case 30:
                        DecimalStrToUint32(pszData, &pEStatus->iDOB);
                        break;
                    case 31:
                        DecimalStrToUint32(pszData, &pEStatus->eDyeFilmManuf);
                        break;
                    case 32:
                        DecimalStrToUint32(pszData, &pEStatus->eDyeFilmProg);
                        break;
                    case 33:
                        pEStatus->iBitFields &= ~(SEM_DEFAULT | SEM_PLATEN);
                        switch (*pszData) {
                            case '0':
                                pEStatus->iBitFields |= SEM_DEFAULT;
                                break;
                            case '1':
                                pEStatus->iBitFields |= SEM_PLATEN;
                                break;
                            default:
                                break;
                        }
                        break;

                    case 34:
                        DecimalStrToUint32(pszData, &pEStatus->iDummy1);
                        break;
                }

                DataID++;

                // Set the data pointer to the character after the ',' or ':'
                pszData = (uint8_t *)pEnduroStatusData;
                break;

            default:
                // Increment the pointer to the Enduro Status buffer
                pEnduroStatusData++;
                break;
        }
    }
}

bool SetUpLUT(PSETTINGS settings) {
    int      i, j, k;
    uint8_t *pGlobalMem      = 0;
    bool     bLoadYaInternal = false;

    TRACE_IN;

    if (bLUTLoaded) {
        TRACE_STR("bLUTLoaded is true, exiting.");
        TRACE_OUT;
        return true;
    }

    if (HELIX_OEM(settings)) {
        TRACE_STR("Printer type is Helix, setting up MAGIOX");
        pGlobalMem = (uint8_t *)&MAGIOX;
    } else if (PRO360_OEM(settings)) {
        TRACE_STR("Printer type is 360, setting up MAGIR2X");
        pGlobalMem = (uint8_t *)&MAGIR2X;
    } else {
        TRACE_STR("Printer type is (unknown), setting up MAGIRY");
        // ya film only for now..
        pGlobalMem      = (uint8_t *)&MAGIRY;
        bLoadYaInternal = true;
    }

    // pglobal mem needs to be set to the colourtable for the correct OEM type
    // at this point
    if (pGlobalMem) {

        memcpy((uint8_t *)&CMYX, pGlobalMem, sizeof(CMYX));
        pGlobalMem += sizeof(CMYX);

        // retrieve the grid  scaling; !!!!! the first 3 words are a header
        // record the first word of which contains the GridDelta
        GridDelta = CMYX[0];
        for (i = 0; i < GridSize; i++) {
            for (j = 0; j < GridSize; j++) {
                for (k = 0; k < GridSize; k++) {

                    memcpy((uint8_t *)&CMYX, pGlobalMem, sizeof(CMYX));

                    Cyan[i][j][k][PERCEPTUAL] =
                            (unsigned char)(CMYX[CYAN] & 0xFF);
                    Cyan[i][j][k][SATURATED] =
                            (unsigned char)(CMYX[CYAN] >> 8 & 0xFF);
                    Cyan[i][j][k][RELATIVE_COLORIMETRIC] =
                            (unsigned char)(CMYX[CYAN] >> 16 & 0xFF);
                    Cyan[i][j][k][ABSOLUTE_COLORIMETRIC] =
                            (unsigned char)(CMYX[CYAN] >> 24 & 0xFF);

                    Magenta[i][j][k][PERCEPTUAL] =
                            (unsigned char)(CMYX[MAGENTA] & 0xFF);
                    Magenta[i][j][k][SATURATED] =
                            (unsigned char)(CMYX[MAGENTA] >> 8 & 0xFF);
                    Magenta[i][j][k][RELATIVE_COLORIMETRIC] =
                            (unsigned char)(CMYX[MAGENTA] >> 16 & 0xFF);
                    Magenta[i][j][k][ABSOLUTE_COLORIMETRIC] =
                            (unsigned char)(CMYX[MAGENTA] >> 24 & 0xFF);

                    Yellow[i][j][k][PERCEPTUAL] =
                            (unsigned char)(CMYX[YELLOW] & 0xFF);
                    Yellow[i][j][k][SATURATED] =
                            (unsigned char)(CMYX[YELLOW] >> 8 & 0xFF);
                    Yellow[i][j][k][RELATIVE_COLORIMETRIC] =
                            (unsigned char)(CMYX[YELLOW] >> 16 & 0xFF);
                    Yellow[i][j][k][ABSOLUTE_COLORIMETRIC] =
                            (unsigned char)(CMYX[YELLOW] >> 24 & 0xFF);

                    pGlobalMem += sizeof(CMYX);
                }
            }
        }

        TRACE("bLoadYaInternal = %s", ShowTrueFalse(bLoadYaInternal));

        if (bLoadYaInternal) {
            // now load up the internals..
            for (i = 0; i < GridSize; i++) {
                for (j = 0; j < GridSize; j++) {
                    for (k = 0; k < GridSize; k += 4) {

                        memcpy((uint8_t *)&CMYX, pGlobalMem, sizeof(CMYX));
                        pGlobalMem += sizeof(CMYX);

                        Cyan[i][j][k][INTERNAL] =
                                (unsigned char)(CMYX[CYAN] & 0xFF);
                        Magenta[i][j][k][INTERNAL] =
                                (unsigned char)(CMYX[MAGENTA] & 0xFF);
                        Yellow[i][j][k][INTERNAL] =
                                (unsigned char)(CMYX[YELLOW] & 0xFF);

                        if (k == 64) {
                            break;
                        }

                        Cyan[i][j][k + 1][INTERNAL] =
                                (unsigned char)(CMYX[CYAN] >> 8 & 0xFF);
                        Cyan[i][j][k + 2][INTERNAL] =
                                (unsigned char)(CMYX[CYAN] >> 16 & 0xFF);
                        Cyan[i][j][k + 3][INTERNAL] =
                                (unsigned char)(CMYX[CYAN] >> 24 & 0xFF);

                        Magenta[i][j][k + 1][INTERNAL] =
                                (unsigned char)(CMYX[MAGENTA] >> 8 & 0xFF);
                        Magenta[i][j][k + 2][INTERNAL] =
                                (unsigned char)(CMYX[MAGENTA] >> 16 & 0xFF);
                        Magenta[i][j][k + 3][INTERNAL] =
                                (unsigned char)(CMYX[MAGENTA] >> 24 & 0xFF);

                        Yellow[i][j][k + 1][INTERNAL] =
                                (unsigned char)(CMYX[YELLOW] >> 8 & 0xFF);
                        Yellow[i][j][k + 2][INTERNAL] =
                                (unsigned char)(CMYX[YELLOW] >> 16 & 0xFF);
                        Yellow[i][j][k + 3][INTERNAL] =
                                (unsigned char)(CMYX[YELLOW] >> 24 & 0xFF);
                    }
                }
            }
        }

        TRACE_STR("LUT Set");
        bLUTLoaded = true;
    } else {
        TRACE_STR("can't open the intents file");
        TRACE_OUT;
        return false;
    }
    TRACE_OUT;
    return true;
}

/*----------------------------------------------------------------------------*/
#ifndef TEST
int main(int argc, char *argv[]) {
    int                 fd = 0; /* File descriptor providing CUPS raster data */
    cups_raster_t *     ras = NULL; /* Raster stream for printing */
    cups_page_header2_t header;     /* CUPS Page header */
    cups_page_header2_t last_header;     /* CUPS Page header */
    int                 page = 0;   /* Current page */

    unsigned char *rasterData = NULL; /* Pointer to raster data buffer */

    /* Copy of original pointer for freeing buffer */
    unsigned char *  originalRasterDataPtr = NULL;
    struct settings_ settings; /* Configuration settings */
    PDEVDATA         pdev;

    char buffer[256];
    int  bytes   = 0;
    int  retries = 2; /* number of retries for status */

    TRACE_IN;

#ifdef RPMBUILD
    void *libCupsImage = NULL; /* Pointer to libCupsImage library */
    void *libCups      = NULL; /* Pointer to libCups library */

    libCups = dlopen("libcups.so", RTLD_NOW | RTLD_GLOBAL);
    if (!libCups) {
        fputs("ERROR: libcups.so load failure\n", stderr);
        return EXIT_FAILURE;
    }

    libCupsImage = dlopen("libcupsimage.so", RTLD_NOW | RTLD_GLOBAL);
    if (!libCupsImage) {
        fputs("ERROR: libcupsimage.so load failure\n", stderr);
        dlclose(libCups);
        return EXIT_FAILURE;
    }

    GET_LIB_FN_OR_EXIT_FAILURE(ppdClose_fn, libCups, "ppdClose");
    GET_LIB_FN_OR_EXIT_FAILURE(ppdFindChoice_fn, libCups, "ppdFindChoice");
    GET_LIB_FN_OR_EXIT_FAILURE(
            ppdFindMarkedChoice_fn, libCups, "ppdFindMarkedChoice");
    GET_LIB_FN_OR_EXIT_FAILURE(ppdFindOption_fn, libCups, "ppdFindOption");
    GET_LIB_FN_OR_EXIT_FAILURE(ppdMarkDefaults_fn, libCups, "ppdMarkDefaults");
    GET_LIB_FN_OR_EXIT_FAILURE(ppdOpenFile_fn, libCups, "ppdOpenFile");
    GET_LIB_FN_OR_EXIT_FAILURE(cupsFreeOptions_fn, libCups, "cupsFreeOptions");
    GET_LIB_FN_OR_EXIT_FAILURE(
            cupsParseOptions_fn, libCups, "cupsParseOptions");
    GET_LIB_FN_OR_EXIT_FAILURE(cupsMarkOptions_fn, libCups, "cupsMarkOptions");
    GET_LIB_FN_OR_EXIT_FAILURE(
            cupsRasterOpen_fn, libCupsImage, "cupsRasterOpen");
    GET_LIB_FN_OR_EXIT_FAILURE(
            cupsRasterReadHeader2_fn, libCupsImage, "cupsRasterReadHeader2");
    GET_LIB_FN_OR_EXIT_FAILURE(
            cupsRasterReadPixels_fn, libCupsImage, "cupsRasterReadPixels");
    GET_LIB_FN_OR_EXIT_FAILURE(
            cupsRasterClose_fn, libCupsImage, "cupsRasterClose");
#endif // defined RPMBUILD

    if (argc < 6 || argc > 7) {
        fputs("ERROR: rastertoultra job-id user title copies options [file]\n",
              stderr);
#ifdef RPMBUILD
        dlclose(libCupsImage);
        dlclose(libCups);
#endif
        return EXIT_FAILURE;
    }

    if (argc == 7) {
        if ((fd = open(argv[6], O_RDONLY)) == -1) {
            perror("ERROR: Unable to open raster file - ");
            sleep(1);
#ifdef RPMBUILD
            dlclose(libCupsImage);
            dlclose(libCups);
#endif
            return EXIT_FAILURE;
        }
    } else {
        fd = 0;
    }
    fflush(stdout);

    pdev = malloc(sizeof(DEVDATA));
    if (pdev == 0) {
        return EXIT_FAILURE;
    }

    memset(pdev, 0, sizeof(DEVDATA));

    initializeSettings(argv[5], &settings, pdev);

    TRACE("settings.modelNumber  = [%u] ", settings.modelNumber);

    ras = CUPSRASTEROPEN(fd, CUPS_RASTER_READ);

    page = 0;

    TRACE("settings.OEM = [%d], settings.Printer = [%d]",
          settings.OEM,
          settings.Printer);

    // retrieve enduro status
    if (settings.OEM != OEM_HELIX && settings.OEM != OEM_PRO360) {
        for (; retries; retries--) {
            printf("\x01,REQ,INF,\x1C\x03");

            fflush(stdout);
            memset(buffer, 0, sizeof(buffer));

            bytes = cupsBackChannelRead(buffer, sizeof(buffer), 2.0);
            if (bytes) {
                TRACE("%d got something? %s", bytes, buffer);
                if (strncmp("STA$$", (char *)&buffer[1], 5) == 0) {
                    ParseEnduroStatus(&settings, &buffer[6]);
                }

                TRACE("settings.sModel [%s]", settings.es.sModel);
                TRACE("settings.iPrintHeadPosn [%u]\n",
                      settings.es.iPrintHeadPosn);
                break;
            } else {
                continue;
            }
        }
    }

    // initialise LUT
    SetUpLUT(&settings);

    while (CUPSRASTERREADHEADER2(ras, &header)) {
        if ((header.cupsHeight == 0) || (header.cupsBytesPerLine == 0)) {
            break;
        }

        if (rasterData == NULL) {
            rasterData = malloc(header.cupsBytesPerLine);
            if (rasterData == NULL) {
                CLEANUP;
                return EXIT_FAILURE;
            }

            // used to later free the memory
            originalRasterDataPtr = rasterData;
        }
        TRACE("header.Duplex = %d", header.Duplex);
        TRACE("header.LeadingEdge = %d", header.LeadingEdge);
        page++;
        pdev->iPageNumber = page;

        // output our header data for the page
        pageSetup(pdev, settings, header);

        // put some constraints in here..
        TRACE("header.cupsHeight: %u, header.cupsBytesPerLine:%u",
              header.cupsHeight,
              header.cupsBytesPerLine);
        TRACE("settings.Printer %u", settings.Printer);

        // HERE WE GO!!
        SendPage(pdev, &settings, header, ras, rasterData);
        TRACE("passed SendPage PAGE %u", page);
        endPage(pdev, &settings, header, true);
        last_header = header;
    }

    // If we've got here on an odd page number, and Duplex selected, it means
    // that endPage() didn't send the page as it was awaiting a second page to
    // form the duplex. This didn't happen, so we have to send it now.
    if (page % 2 == 1 && pdev->bDuplex && pdev->bFrontPage) {
       TRACE_STR("CD-124 fix: Sending single page,"
                     "despite duplex being selected");
       pdev->bDuplex = false;
       endPage(pdev, &settings, last_header, false);
    }

    endJob(pdev, settings, header);
    free(pdev->pso32bitSurface);
    free(pdev);

    CLEANUP;

    if (page == 0) {
        fputs("ERROR: No pages found!\n", stderr);
    } else {
        fputs("INFO: Ready to print.\n", stderr);
    }

    return (page == 0) ? EXIT_FAILURE : EXIT_SUCCESS;
}
#endif /* not defined TEST */

// end of rastertoultra.c
