/*
 * Magicard Ltd
 *
 * CUPS Filter
 *
 * [ Linux ]
 * compile cmd: gcc -Wl,-rpath,/usr/lib -Wall -fPIC -O2 -o cmdtoultra
 * cmdtoultra.c -lcupsimage -lcups
 * compile requires cups-devel-1.1.19-13.i386.rpm (version not neccessarily
 * important?)
 * find cups-devel location here:
 * http://rpmfind.net/linux/rpm2html/search.php?query=cups-devel&submit=Search+...&system=&arch=
 *
 * [ Mac OS X ]
 * compile cmd: gcc -Wall -fPIC -o cmdtoultra cmdtoultra.c -lcupsimage -lcups
 * -arch ppc -arch i386 -DMACOSX
 * compile requires cupsddk-1.2.3.dmg (version not neccessarily important?)
 * find cupsddk location here: http://www.cups.org/software.php
 */

/*
 * Copyright (C) 2018 Magicard Ltd
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/*
 * Include necessary headers...
 */

#include <stdlib.h>
#include <stdio.h>
#include <strings.h>
#include <fcntl.h>
#include <cups/backend.h>

#include "cmddefs.h"
#include "comms.h"
#include "gndefs.h"
#include "magigen.h"
#include "pr2comms.h"
#include "debug_utils.h"
#include "crc32.h"
#include "magicard_cmds.h"

static const char* CUPS_CMD_HEADER = "#CUPS-COMMAND\n";

/******************************************************************************
 * write_cmd
 *     Write the supplied command to stdout complete with a crc32 checksum.
 *
 * @param cmd  - The complete command xml string to send.
 *               Must be NULL terminated.
 *
 * @returns    - 0 on success, else the number of bytes written.
 *****************************************************************************/
static int write_cmd(const char* cmd) {

   size_t written = 0;
   uint32_t chksum = 0xffffffffu;

   fflush(stdout);

   // include the NULL in the crc
   crc32update(&chksum, cmd, strlen(cmd)+1);

   // write out the command, including the NULL
   written = write(STDOUT_FILENO, cmd, strlen(cmd)+1);

   // finalise the checksum
   chksum ^= 0xffffffffu;
   TRACE("chksum = %x", chksum);

   // Finish the command with the checksum
   written += write(STDOUT_FILENO, &chksum, sizeof(chksum));
   TRACE("%zu bytes written to stdout", written);

   fflush(stdout);

   if (written == strlen(cmd)+1 + sizeof(chksum)) {
      return 0;
   } else {
      return written;
   }
}

/*****************************************************************************/
static int start_clean_routine() {
   const char* base_cmd = "<?xml version=\"1.0\"; encoding=\"UTF-8\"?>\
<prn><binary-size>0</binary-size>\
<time>%d</time><start-clean-rollers/>\
</prn>\0";
   char cmd[512] = {'\0'};
   time_t now = 0;


   // add the dynamic fields of time and test-id
   time(&now);
   sprintf(cmd, base_cmd, now);
   TRACE("COMMAND: %s", cmd);

   write_cmd(cmd);
   return 0;
}

/*****************************************************************************/
static int print_test_card(const char* test_id) {
   const char* base_cmd = "<?xml version=\"1.0\"; encoding=\"UTF-8\"?>\
<prn><binary-size>0</binary-size>\
<time>%d</time><test-print><test-id>%s</test-id></test-print>\
</prn>\0";
   char cmd[512] = {'\0'};
   time_t now = 0;

   // add the dynamic fields of time and test-id
   time(&now);
   sprintf(cmd, base_cmd, now, test_id == NULL?"setup":test_id);
   TRACE("COMMAND: %s", cmd);

   write_cmd(cmd);
   return 0;
}

/******************************************************************************
 * read_cmd_file
 *     This function opens the specified command file and extracts the
 *     requested command from it, returning the appropriate magicard_cmd_t.
 *
 * @Errors - If the command isn't implemented, CMD_UNKNOWN will be returned.
 *           If there is a problem opening or reading the file, or the format
 *           of the command file isn't as expected, CMD_ERROR will be returned.
 *
 * @ref    - https://www.cups.org/doc/spec-command.html
 *
 *****************************************************************************/
static magicard_cmd_t read_cmd_file(const char* cmd_file) {
   magicard_cmd_t ret = CMD_ERROR;
   FILE* f = fopen(cmd_file, "r");
   TRACE_IN;

   if (f != NULL) {
      char* line = NULL;
      size_t line_len = 0;
      size_t line_num = 0;
      ssize_t nchars = 0;

      while ((nchars = getline(&line, &line_len,f)) != -1) {
         line_num++;
         TRACE("Read %zd chars", nchars);

         /* Handle the first line */
         /* we can cast nchars, as we can be certain it's not a negative number
          *  due to the check in the while loop condition.
          */
         if(line_num == 1) {
            if ((size_t)nchars == strlen(CUPS_CMD_HEADER)) {
               TRACE("First line read OK (%zd chars)", nchars);
               if (strcmp(CUPS_CMD_HEADER, line) == 0){
                  TRACE("%s", "Valid header");
                  continue;
               } else {
                  ERR("First line not '%s' - '%s'", CUPS_CMD_HEADER, line);
               }
            } else {
               ERR("CUPS cmd file first line incorrect length: \
                     expected %zd, got %zd", strlen(CUPS_CMD_HEADER), nchars);
            }
            break;
         }

         // Skip comment lines
         if (line[0] == '#') {
            continue;
         }

         if (strcasecmp(line, "Clean all\n") == 0 ) {
            TRACE("Clean command found on line %zu", line_num);
            ret = CMD_CLEAN;
         } else if (strcasecmp(line, "PrintSelfTestPage\n") == 0 ) {
            TRACE("Print internal test card command found on line %zu",
                   line_num);
            ret = CMD_PRINT_INTERNAL_TEST_CARD;
         } else {
            TRACE("Unkown cmd '%s' found on line %zu", line, line_num);
            ret = CMD_UNKNOWN;
         }
         break;
      } // while

      // getline allocates the memory for the line buffer,
      // free it now we're done.
      free(line);
      fclose(f);
   } else {
      ERR("Unable to open CUPS cmd file '%s'.", cmd_file);
   }

   TRACE_OUT;
   return ret;
}

/*****************************************************************************/
static int open_ppd_file(ppd_file_t** ppd_file) {
   TRACE_IN;
   *ppd_file = PPDOPENFILE(getenv("PPD"));
   PPDMARKDEFAULTS(*ppd_file);
   TRACE_OUT;
   return 0;
}

/*****************************************************************************/
int main(int argc, char *argv[]) {
    magicard_cmd_t cmd = CMD_UNKNOWN;

#ifdef RPMBUILD
    /* Pointer to libCupsImage library */
    void *libCupsImage = NULL;

    /* Pointer to libCups library */
    void *libCups = NULL;

    libCups = dlopen("libcups.so", RTLD_NOW | RTLD_GLOBAL);
    if (!libCups) {
        fputs("ERROR: libcups.so load failure\n", stderr);
        return EXIT_FAILURE;
    }

    libCupsImage = dlopen("libcupsimage.so", RTLD_NOW | RTLD_GLOBAL);
    if (!libCupsImage) {
        fputs("ERROR: libcupsimage.so load failure\n", stderr);
        dlclose(libCups);
        return EXIT_FAILURE;
    }

    GET_LIB_FN_OR_EXIT_FAILURE(ppdClose_fn, libCups, "ppdClose");
    GET_LIB_FN_OR_EXIT_FAILURE(ppdFindChoice_fn, libCups, "ppdFindChoice");
    GET_LIB_FN_OR_EXIT_FAILURE(
            ppdFindMarkedChoice_fn, libCups, "ppdFindMarkedChoice");
    GET_LIB_FN_OR_EXIT_FAILURE(ppdFindOption_fn, libCups, "ppdFindOption");
    GET_LIB_FN_OR_EXIT_FAILURE(ppdMarkDefaults_fn, libCups, "ppdMarkDefaults");
    GET_LIB_FN_OR_EXIT_FAILURE(ppdOpenFile_fn, libCups, "ppdOpenFile");
    GET_LIB_FN_OR_EXIT_FAILURE(cupsFreeOptions_fn, libCups, "cupsFreeOptions");
    GET_LIB_FN_OR_EXIT_FAILURE(
            cupsParseOptions_fn, libCups, "cupsParseOptions");
    GET_LIB_FN_OR_EXIT_FAILURE(cupsMarkOptions_fn, libCups, "cupsMarkOptions");
    GET_LIB_FN_OR_EXIT_FAILURE(
            cupsRasterOpen_fn, libCupsImage, "cupsRasterOpen");
    GET_LIB_FN_OR_EXIT_FAILURE(
            cupsRasterReadHeader2_fn, libCupsImage, "cupsRasterReadHeader2");
    GET_LIB_FN_OR_EXIT_FAILURE(
            cupsRasterReadPixels_fn, libCupsImage, "cupsRasterReadPixels");
    GET_LIB_FN_OR_EXIT_FAILURE(
            cupsRasterClose_fn, libCupsImage, "cupsRasterClose");
#endif

    if (argc < 6 || argc > 7) {
        fputs("ERROR: cmdtoultra job-id user title copies options [file]\n",
              stderr);
#ifdef RPMBUILD
        dlclose(libCupsImage);
        dlclose(libCups);
#endif
        return EXIT_FAILURE;
    }

    if (argc == 7) {
        cmd = read_cmd_file(argv[6]);
        TRACE("cmd = %d", cmd);
    }
    fflush(stdout);

    // Check the PPD file name passed to us, and read it.
    if (strlen(argv[3])) {
       ppd_file_t *ppd;
       if (open_ppd_file(&ppd) == 0) {
        if (cmd == CMD_CLEAN) {
           if(ppd->model_number >= PRO360) {
              start_clean_routine();
           } else {
              // send a command to the printer to clean the print head
              printf("\x05\x05\x05\x05\x05\x05\x05\x05\x01,REQ,CLN,\x1C\x03");
              fflush(stdout);
              return EXIT_SUCCESS;
           }
        }

        if (cmd == CMD_PRINT_INTERNAL_TEST_CARD) {
           if(ppd->model_number >= PRO360) {
              print_test_card("setup");
           } else {
              TRACE("%s", "Sending legacy test-page");
              // send a command to the printer to request a test page
              printf("\x05\x05\x05\x05\x05\x05\x05\x05\x01,REQ,TST,\x1C\x03");
              fflush(stdout);
              return EXIT_SUCCESS;
           }
        }
        PPDCLOSE(ppd);
       } else {
          ERR("%s", "Failed to open PPD file.");
       }
    }
    TRACE_OUT;
    return EXIT_SUCCESS;
}

// end of cmdtoultra.c
